import { fileExists } from './src/utils/file.js';
import express from 'express';
import session from 'express-session';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';
import crypto from 'crypto';
import path from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';
import fs from 'fs/promises'; // Using fs/promises for async file operations
import http from 'http';
import globals from './src/middlewares/globals.js';
import maintenance from './src/middlewares/maintenance.js';
import languageMiddleware from './src/middlewares/language.js';
import templateResolver from './src/middlewares/template_resolver.js';
import { initWebSocketServer, closeWebSocketServer } from './src/utils/websocket.js';
import { consoleLog } from './src/utils/logger.js';
import { closePool } from './src/utils/mysql.js';
import CacheUtils from './src/utils/cache.js';
import passport from 'passport';
import { initializePassport } from './src/utils/passport.js';
import i18n from './src/utils/i18n.js';

const envFileExist = await fileExists('.env');

if (!envFileExist) {
  consoleLog('error', 'The .env file does not exist. Please create it with appropriate configurations.');
  process.exit(1); // Exit the process
}

dotenv.config();

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const app = express();

// CSRF protection now uses stateless tokens (no longer needs csrf library)

// Stateless CSRF token functions
const generateStatelessCSRFToken = (req) => {
  const timestamp = Date.now();
  const clientId = req.ip || 'unknown';
  const userAgent = req.get('User-Agent') || 'unknown';

  // Create a unique identifier for this client/session
  const clientHash = crypto
    .createHash('sha256')
    .update(clientId + userAgent)
    .digest('hex')
    .substring(0, 16);

  // Generate random token
  const randomToken = crypto.randomBytes(24).toString('hex');

  // Create signature with server secret, timestamp, and client info
  const signature = crypto
    .createHmac('sha256', process.env.SESSION_SECRET + 'csrf-stateless')
    .update(`${randomToken}:${timestamp}:${clientHash}`)
    .digest('hex');

  // Combine token parts
  return `${randomToken}:${timestamp}:${clientHash}:${signature}`;
};

const verifyStatelessCSRFToken = (token, req) => {
  if (!token || typeof token !== 'string') return false;

  const parts = token.split(':');
  if (parts.length !== 4) return false;

  const [randomToken, timestamp, clientHash, signature] = parts;

  // Verify timestamp (tokens valid for 2 hours)
  const now = Date.now();
  const tokenAge = now - parseInt(timestamp);
  if (tokenAge > 2 * 60 * 60 * 1000 || tokenAge < 0) return false;

  // Recreate client hash
  const clientId = req.ip || 'unknown';
  const userAgent = req.get('User-Agent') || 'unknown';
  const expectedClientHash = crypto
    .createHash('sha256')
    .update(clientId + userAgent)
    .digest('hex')
    .substring(0, 16);

  // Verify client hash matches
  if (clientHash !== expectedClientHash) return false;

  // Verify signature
  const expectedSignature = crypto
    .createHmac('sha256', process.env.SESSION_SECRET + 'csrf-stateless')
    .update(`${randomToken}:${timestamp}:${clientHash}`)
    .digest('hex');

  return signature === expectedSignature;
};

// Security middleware - Helmet for security headers
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"], // Default to self
      scriptSrc: [
        "*", 
        "'self'",
        "'unsafe-inline'",
        "'unsafe-eval'", // Required for EmulatorJS and gaming functionality
        "'unsafe-hashes'", // Required for inline event handlers (onclick, etc.)
        "blob:",
        "data:",
        "'unsafe-inline'"
      ], // Allow all script sources and inline scripts
      scriptSrcAttr: ["*", "'unsafe-inline'"], // Allow all script attributes and inline handlers
      styleSrc: ["*", "'unsafe-inline'"], // Allow all style sources and inline styles
      imgSrc: ["*", "'self'", "data:", "blob:"], // Allow all image sources for game assets
      connectSrc: [
        "*",
        "'self'",
        "blob:",
        "data:"
      ], // Allow all connection sources
      fontSrc: ["*"], // Allow all font sources
      objectSrc: ["'none'"], // Disallow all object sources
      mediaSrc: ["*", "'self'", "blob:", "data:"], // Allow all media sources
      frameSrc: ["*"], // Allow all iframe sources
      frameAncestors: [
        "'self'"
      ], // Only allow framing by same origin
      workerSrc: ["'self'", "blob:"], // Allow workers from self and blob
      upgradeInsecureRequests: process.env.ENVIRONMENT === 'production' ? [] : null // Enable in production only
    }
  },
  crossOriginEmbedderPolicy: false, // Disabled for gaming content compatibility
  crossOriginResourcePolicy: { policy: "cross-origin" },
  crossOriginOpenerPolicy: false, // Disabled to prevent the warning on non-HTTPS origins
  originAgentCluster: false // Disabled to prevent agent cluster warnings
}));

// Additional security headers for development to prevent console warnings
if (process.env.ENVIRONMENT !== 'production') {
  app.use((req, res, next) => {
    // Prevent Cross-Origin-Opener-Policy warnings on HTTP development
    res.removeHeader('Cross-Origin-Opener-Policy');
    // Prevent Origin-Agent-Cluster warnings 
    res.removeHeader('Origin-Agent-Cluster');
    next();
  });
}

// Rate limiting configuration from environment variables
const limiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS) || 15 * 60 * 1000, // Default: 15 minutes
  max: parseInt(process.env.RATE_LIMIT_MAX_REQUESTS) || 1000, // Default: 1000 requests per window
  message: (req) => ({
    error: i18n.translateSync('errors.rate_limit', {}, req.language?.current || 'en'),
    code: 'RATE_LIMIT_EXCEEDED'
  }),
  standardHeaders: true,
  legacyHeaders: false,
});

const authLimiter = rateLimit({
  windowMs: parseInt(process.env.AUTH_RATE_LIMIT_WINDOW_MS) || 15 * 60 * 1000, // Default: 15 minutes
  max: parseInt(process.env.AUTH_RATE_LIMIT_MAX) || 5, // Default: 5 attempts per window
  message: (req) => ({
    error: i18n.translateSync('errors.auth_rate_limit', {}, req.language?.current || 'en'),
    code: 'AUTH_RATE_LIMIT_EXCEEDED'
  }),
  skipSuccessfulRequests: true,
  standardHeaders: true,
  legacyHeaders: false,
});

// Apply rate limiting
app.use('/requests/', limiter); // API endpoints are under /requests/
app.use('/auth/login', authLimiter);
app.use('/auth/register', authLimiter);
app.use('/auth/forgot-password', authLimiter);
app.use('/auth/reset-password', authLimiter);

// CORS configuration with origin restrictions
const corsOptions = {
  origin: function (origin, callback) {
    // Allow requests with no origin (mobile apps, etc.)
    if (!origin) return callback(null, true);

    // In production, restrict to your domains
    const allowedOrigins = process.env.ALLOWED_ORIGINS
      ? process.env.ALLOWED_ORIGINS.split(',')
      : ['http://localhost:3000', 'http://localhost:8080']; // Default for development

    if (allowedOrigins.indexOf(origin) !== -1 || process.env.ENVIRONMENT !== 'production') {
      callback(null, true);
    } else {
      callback(new Error('Not allowed by CORS'));
    }
  },
  credentials: true,
  optionsSuccessStatus: 200,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With', 'X-CSRF-Token']
};

app.use(cors(corsOptions));

// Body parsing middleware
app.use(express.urlencoded({
  extended: true,
  limit: process.env.BODY_SIZE_LIMIT || '100mb' // Configurable body size limit
}));
app.use(express.json({
  limit: process.env.JSON_SIZE_LIMIT || '10mb' // Configurable JSON body size limit
}));

// Static file serving
app.use(express.static(path.join(__dirname, 'public')));
app.use('/uploads', express.static(path.join(__dirname, 'uploads')));

// Serve documentation directory only in development environment
if (process.env.ENVIRONMENT === 'development') {
    app.use('/documentation', express.static(path.join(__dirname, 'documentation')));
}

// Session security: secure cookies enabled in production by default
// Can be disabled with DISABLE_SECURE_COOKIES=true for HTTPS troubleshooting
// WARNING: Disabling secure cookies in production reduces security
// Secure session configuration (optimized for stateless CSRF)
app.use(session({
  secret: process.env.SESSION_SECRET,
  resave: false,
  saveUninitialized: false, // Don't create sessions unnecessarily (CSRF is now stateless)
  rolling: true, // Extend session expiration on activity
  unset: 'destroy', // Clean up properly when session is unset
  name: 'arcade.sid', // Change default session name
  cookie: {
    secure: process.env.ENVIRONMENT === 'production' && process.env.DISABLE_SECURE_COOKIES !== 'true',
    httpOnly: true, // Prevent XSS attacks
    maxAge: 24 * 60 * 60 * 1000, // 24 hours
    sameSite: 'lax' // CSRF protection
  }
}));

// Initialize Passport.js for OAuth
app.use(passport.initialize());
app.use(passport.session());
await initializePassport();

// CSRF token generation middleware (stateless tokens)
app.use((req, res, next) => {
  try {
    // Generate stateless CSRF token for this request
    const token = generateStatelessCSRFToken(req);

    if (!token) {
      // Fallback token generation if primary fails
      consoleLog('warn', `[CSRF] Primary token generation failed, using fallback for ${req.ip}`);
      const fallbackToken = crypto.randomBytes(32).toString('hex') + ':fallback';
      res.locals.csrfToken = fallbackToken;
    } else {
      res.locals.csrfToken = token;
    }
  } catch (error) {
    // Emergency fallback - should never happen but better safe than sorry
    consoleLog('error', `[CSRF] Token generation error: ${error.message}`);
    const emergencyToken = crypto.randomBytes(32).toString('hex') + ':emergency';
    res.locals.csrfToken = emergencyToken;
  }

  next();
});

// CSRF verification middleware function (to be used in routes)
const verifyCSRF = (req, res, next) => {
  // Skip verification for GET requests and specific paths
  if (req.method === 'GET' ||
      req.path.startsWith('/uploads/') ||
      req.path.startsWith('/auth/facebook') ||
      req.path.startsWith('/auth/google')) {
    return next();
  }

  try {
    const providedToken = req.body._csrf || req.query._csrf || req.headers['x-csrf-token'];

    if (!providedToken) {
      // Log for monitoring without exposing sensitive details
      consoleLog('warn', `[CSRF] Missing token - ${req.method} ${req.path} from ${req.ip} at ${new Date().toISOString()}`);
      const error = new Error('CSRF token missing');
      error.code = 'EBADCSRFTOKEN';
      throw error;
    }

    // Check for fallback/emergency tokens and reject them gracefully
    if (providedToken.endsWith(':fallback') || providedToken.endsWith(':emergency')) {
      consoleLog('warn', `[CSRF] Fallback/emergency token rejected - ${req.method} ${req.path} from ${req.ip}`);
      const error = new Error('CSRF token requires page refresh');
      error.code = 'EBADCSRFTOKEN';
      throw error;
    }

    if (!verifyStatelessCSRFToken(providedToken, req)) {
      // Log verification failure for monitoring
      consoleLog('warn', `[CSRF] Token verification failed - ${req.method} ${req.path} from ${req.ip} at ${new Date().toISOString()}`);
      const error = new Error('CSRF token invalid');
      error.code = 'EBADCSRFTOKEN';
      throw error;
    }

    // Token is valid, proceed
    next();
  } catch (err) {
    if (err.code === 'EBADCSRFTOKEN') {
      // Enhanced error response with optional debug info
      const response = {
        error: i18n.translateSync('errors.csrf_expired', {}, req.language?.current || 'en'),
        code: 'CSRF_INVALID'
      };

      // Add debug info in development mode only
      if (process.env.ENVIRONMENT === 'development') {
        response.debug = err.message;
        response.timestamp = new Date().toISOString();
      }

      res.status(403).json(response);
    } else {
      next(err);
    }
  }
};

// Make CSRF verification available to routes
app.use((req, res, next) => {
  req.verifyCSRF = verifyCSRF;
  next();
});

// Application middleware
app.use(languageMiddleware);
app.use(globals);
app.use(templateResolver);


app.use(maintenance);
app.set('view engine', 'ejs');
// Trust proxy configuration - be specific to avoid rate limiting bypass
app.set('trust proxy', process.env.ENVIRONMENT === 'production' ? 1 : false);

// Dynamically include routes
const routesDir = path.join(__dirname, 'src', 'routers');
const files = await fs.readdir(routesDir);

for (const file of files) {
  if (file != 'default.js' && file.endsWith('.js')) {
    const routeModule = await import(path.join(routesDir, file));
    const routePath = `/${path.basename(file, '.js')}`;
    consoleLog('server', `Adding route: ${routePath}`);
    app.use(routePath, routeModule.default);
  } else {
    // Root route
    const rootModule = await import(path.join(routesDir, 'default.js'));
    app.use('/', rootModule.default);
  }
}

app.all("*", (req, res) => {
  const pageData = {
    page: "errors",
    title: "404",
    description: i18n.translateSync('errors.404_description', {}, req.language?.current || 'en')
  };

  res.render("pages/errors/404", pageData);
});

// Create HTTP server
const server = http.createServer(app);

// Initialize WebSocket server
initWebSocketServer(server);

// Start the server
server.listen(process.env.SERVER_PORT, () => {
  consoleLog('server', `Server is running on port ${process.env.SERVER_PORT}`);
});

// Graceful shutdown handlers
async function gracefulShutdown(signal) {
  consoleLog('server', `Received ${signal}. Starting graceful shutdown...`);
  
  try {
    // Stop accepting new connections
    server.close(async () => {
      consoleLog('server', 'HTTP server closed');
      
      // Close WebSocket connections
      closeWebSocketServer();
      
      // Clean up cache instances
      await CacheUtils.cleanupUnusedCaches(0); // Clean all caches on shutdown
      
      // Close database connection pool
      await closePool();
      
      consoleLog('server', 'Graceful shutdown completed');
      process.exit(0);
    });
    
    // Force exit after 3 seconds if graceful shutdown fails
    setTimeout(() => {
      consoleLog('error', 'Graceful shutdown timed out, forcing exit');
      process.exit(1);
    }, 3000);
    
  } catch (error) {
    consoleLog('error', 'Error during graceful shutdown', { error: error.message });
    process.exit(1);
  }
}

// Handle different shutdown signals
process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
process.on('SIGINT', () => gracefulShutdown('SIGINT'));
process.on('SIGUSR2', () => gracefulShutdown('SIGUSR2')); // Nodemon restart

// Handle uncaught exceptions
process.on('uncaughtException', (error) => {
  consoleLog('error', 'Uncaught Exception', { error: error.message, stack: error.stack });
  gracefulShutdown('uncaughtException');
});

process.on('unhandledRejection', (reason, promise) => {
  consoleLog('error', 'Unhandled Rejection', { reason, promise });
  gracefulShutdown('unhandledRejection');
});