#!/bin/bash

# =======================================================
# TITAN SYSTEMS ARCADE - Fully Automated Ubuntu 24.04 Setup
# =======================================================
# This script provides complete automation for:
# - MySQL 8.0 installation and configuration
# - Node.js 20 LTS installation
# - Database creation and user setup
# - Arcade platform deployment and configuration
# - Service setup and startup
# =======================================================

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
BOLD='\033[1m'
NC='\033[0m' # No Color

# Logging configuration
INSTALL_LOG="install.log"
LOG_MAX_SIZE=10485760  # 10MB

# Initialize logging system
setup_logging() {
    # Create log file with timestamp header
    echo "========================================" > "$INSTALL_LOG"
    echo "TITANS ARCADE INSTALLATION LOG" >> "$INSTALL_LOG"
    echo "Started: $(date)" >> "$INSTALL_LOG"
    echo "Script: $0" >> "$INSTALL_LOG"
    echo "Arguments: $*" >> "$INSTALL_LOG"
    echo "========================================" >> "$INSTALL_LOG"
    echo "" >> "$INSTALL_LOG"

    # Check if log file is getting too large
    if [[ -f "$INSTALL_LOG" ]] && [[ $(stat -f%z "$INSTALL_LOG" 2>/dev/null || stat -c%s "$INSTALL_LOG" 2>/dev/null || echo 0) -gt $LOG_MAX_SIZE ]]; then
        mv "$INSTALL_LOG" "${INSTALL_LOG}.old"
        setup_logging
        return
    fi

    echo -e "${BLUE}[INFO]${NC} Logging system initialized - output will be saved to $INSTALL_LOG"
}

# Log functions with dual output (console + file)
log_info() {
    local msg="$1"
    echo -e "${BLUE}[INFO]${NC} $msg"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [INFO] $msg" >> "$INSTALL_LOG"
}

log_success() {
    local msg="$1"
    echo -e "${GREEN}[SUCCESS]${NC} $msg"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [SUCCESS] $msg" >> "$INSTALL_LOG"
}

log_warning() {
    local msg="$1"
    echo -e "${YELLOW}[WARNING]${NC} $msg"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [WARNING] $msg" >> "$INSTALL_LOG"
}

log_error() {
    local msg="$1"
    echo -e "${RED}[ERROR]${NC} $msg"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [ERROR] $msg" >> "$INSTALL_LOG"
}

log_header() {
    echo -e "\n${CYAN}=================================${NC}"
    echo -e "${CYAN} $1 ${NC}"
    echo -e "${CYAN}=================================${NC}\n"
}

log_highlight() {
    echo -e "\n${MAGENTA}═══════════════════════════════════════════════════════════════════${NC}"
    echo -e "${MAGENTA}  $1  ${NC}"
    echo -e "${MAGENTA}═══════════════════════════════════════════════════════════════════${NC}\n"
}

# Loading animation functions
show_loading() {
    local msg="$1"
    local pid=$!
    local delay=0.15
    local spinstr='|/-\'
    echo -n -e "${YELLOW}[WAIT]${NC} $msg "

    while [ "$(ps a | awk '{print $1}' | grep $pid)" ]; do
        local temp=${spinstr#?}
        printf " [%c]  " "$spinstr"
        local spinstr=$temp${spinstr%"$temp"}
        sleep $delay
        printf "\b\b\b\b\b\b"
    done
    printf "    \b\b\b\b"
    echo ""
}

log_wait() {
    local msg="$1"
    echo -e "${YELLOW}[WAIT]${NC} $msg - ${CYAN}Please wait, this may take a few minutes...${NC}"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [WAIT] $msg" >> "$INSTALL_LOG"
}

log_progress() {
    local msg="$1"
    echo -e "${BLUE}[PROGRESS]${NC} $msg"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [PROGRESS] $msg" >> "$INSTALL_LOG"
}

# Configuration variables - will be auto-generated
SITE_NAME="Arcade Platform"
SITE_DESCRIPTION="Ultimate Gaming Platform"
PURCHASE_CODE=""
DB_HOST="localhost"
DB_NAME="arcade"
ADMIN_FIRST_NAME="System"
ADMIN_LAST_NAME="Administrator"
ADMIN_USERNAME="admin"
SERVER_PORT="3000"  # App runs on 3000, Nginx proxies from 80/443
DOMAIN=""
SITE_URL=""
ADMIN_EMAIL=""
# These will be auto-generated
MYSQL_ROOT_PASSWORD=""
DB_USERNAME=""
DB_PASSWORD=""
ADMIN_PASSWORD=""

# Domain handling variables
CANONICAL_DOMAIN=""      # Always non-www version (primary URL)
WWW_DOMAIN=""           # www version (empty for subdomains)
DOMAIN_TYPE=""          # "root", "www_input", "subdomain"
NEEDS_WWW_REDIRECT=false # Whether to setup www→non-www redirect

# Generate secure random password
generate_password() {
    local length=${1:-16}
    openssl rand -base64 32 | tr -d "=+/" | cut -c1-$length
}

# Generate secure random string for usernames
generate_username() {
    echo "user$(openssl rand -hex 4)"
}

# Auto-generate all credentials
generate_credentials() {
    log_header "GENERATING SECURE CREDENTIALS"
    
    # Auto-generate passwords
    MYSQL_ROOT_PASSWORD=$(generate_password 20)
    DB_PASSWORD=$(generate_password 16)
    ADMIN_PASSWORD=$(generate_password 12)
    DB_USERNAME=$(generate_username)
    
    log_success "All credentials generated automatically"
    log_info "MySQL root password: 20 characters"
    log_info "Database password: 16 characters"
    log_info "Admin password: 12 characters"
    log_info "Database username: $DB_USERNAME"
}

# Validate and process domain input with www handling
validate_and_process_domain() {
    local input_domain="$1"

    # Clean input (remove protocols, trailing slashes, convert to lowercase)
    input_domain=$(echo "$input_domain" | sed 's|^https\?://||' | sed 's|/.*||' | tr '[:upper:]' '[:lower:]')

    # Reject empty input
    if [[ -z "$input_domain" ]]; then
        log_error "Domain name cannot be empty"
        return 1
    fi

    # Reject IP addresses (IPv4 and IPv6)
    if [[ "$input_domain" =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]] ||
       [[ "$input_domain" =~ ^[0-9a-fA-F:]+:[0-9a-fA-F:]*$ ]]; then
        log_error "IP addresses are not allowed as domain names"
        log_info "SSL certificates require proper domain names, not IP addresses"
        log_info "Please use a domain name like: example.com or subdomain.example.com"
        return 1
    fi

    # Validate domain format (basic DNS-safe characters)
    if [[ ! "$input_domain" =~ ^[a-z0-9]([a-z0-9-]*[a-z0-9])?(\.[a-z0-9]([a-z0-9-]*[a-z0-9])?)*\.[a-z]{2,}$ ]]; then
        log_error "Invalid domain format: $input_domain"
        log_info "Domain must be a valid format like: example.com or subdomain.example.com"
        return 1
    fi

    # Count dots to identify domain type
    local dot_count=$(echo "$input_domain" | tr -cd '.' | wc -c)

    if [[ "$input_domain" =~ ^www\. ]]; then
        # Input: www.example.com → Canonical: example.com
        DOMAIN_TYPE="www_input"
        CANONICAL_DOMAIN="${input_domain#www.}"
        WWW_DOMAIN="$input_domain"
        NEEDS_WWW_REDIRECT=true
        log_info "WWW domain detected: $input_domain → canonical: $CANONICAL_DOMAIN"
    elif [[ $dot_count -eq 1 ]]; then
        # Input: example.com → Add www redirect
        DOMAIN_TYPE="root"
        CANONICAL_DOMAIN="$input_domain"
        WWW_DOMAIN="www.$input_domain"
        NEEDS_WWW_REDIRECT=true
        log_info "Root domain detected: $input_domain → will redirect www.$input_domain"
    else
        # Input: api.example.com → Subdomain, no www
        DOMAIN_TYPE="subdomain"
        CANONICAL_DOMAIN="$input_domain"
        WWW_DOMAIN=""
        NEEDS_WWW_REDIRECT=false
        log_info "Subdomain detected: $input_domain → no www redirect needed"
    fi

    # Set final domain variable for backward compatibility
    DOMAIN="$CANONICAL_DOMAIN"

    return 0
}

# Get domain name from user
get_domain() {
    local input_domain

    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${MAGENTA}                                    🌐 DOMAIN SETUP REQUIRED 🌐${NC}"
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}⚠️  USER INPUT REQUIRED - PLEASE PROVIDE YOUR DOMAIN NAME ⚠️${NC}"
    echo ""
    echo -e "${YELLOW}Please enter your domain name for SSL certificate:${NC}"
    echo -e "${YELLOW}This will be used to generate your SSL certificate via Let's Encrypt${NC}"
    echo -e "${GREEN}Examples: ${CYAN}example.com${NC} or ${CYAN}subdomain.example.com${NC}"
    echo -e "${RED}Note: IP addresses are not allowed${NC}"
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"

    while true; do
        echo -en "${CYAN}${BOLD}>>> Enter your domain name: ${NC}"
        read input_domain

        if validate_and_process_domain "$input_domain"; then
            break
        else
            echo ""
            log_warning "Please try again with a valid domain name"
            echo ""
        fi
    done

    # Generate site URL and admin email based on canonical domain
    SITE_URL="https://${CANONICAL_DOMAIN}"
    ADMIN_EMAIL="admin@${CANONICAL_DOMAIN}"

    echo ""
    log_success "Domain configuration:"
    log_info "  Primary URL: $SITE_URL"
    log_info "  Admin email: $ADMIN_EMAIL"
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        log_info "  WWW redirect: https://$WWW_DOMAIN → https://$CANONICAL_DOMAIN"
    fi
    echo ""
}

# Get purchase code from user
get_purchase_code() {
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${MAGENTA}                                  🔑 LICENSE VERIFICATION REQUIRED 🔑${NC}"
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}⚠️  USER INPUT REQUIRED - PLEASE PROVIDE YOUR PURCHASE CODE ⚠️${NC}"
    echo ""
    echo -e "${YELLOW}Please enter your purchase code for license verification:${NC}"
    echo -e "${YELLOW}Purchase codes contain letters, numbers, and hyphens only${NC}"
    echo -e "${GREEN}Example: ${CYAN}ABC123-DEF456-GHI789${NC}"
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -en "${CYAN}${BOLD}>>> Enter your purchase code: ${NC}"
    read PURCHASE_CODE

    if [[ -z "$PURCHASE_CODE" ]]; then
        echo ""
        log_error "Purchase code is required for license verification"
        exit 1
    fi

    # Validate purchase code format (alphanumeric and hyphens only)
    if [[ ! "$PURCHASE_CODE" =~ ^[a-zA-Z0-9-]+$ ]]; then
        echo ""
        log_error "Invalid purchase code format"
        log_info "Purchase code must contain only letters, numbers, and hyphens"
        exit 1
    fi

    echo ""
    log_success "Purchase code accepted: ${PURCHASE_CODE:0:10}..."
    echo ""
}

# Parse command line arguments (simplified)
parse_arguments() {
    while [[ $# -gt 0 ]]; do
        case $1 in
            --domain)
                if validate_and_process_domain "$2"; then
                    SITE_URL="https://${CANONICAL_DOMAIN}"
                    ADMIN_EMAIL="admin@${CANONICAL_DOMAIN}"
                    shift 2
                else
                    log_error "Invalid domain provided via command line: $2"
                    exit 1
                fi
                ;;
            --purchase-code)
                PURCHASE_CODE="$2"
                shift 2
                ;;
            --help|-h)
                show_help
                exit 0
                ;;
            *)
                log_error "Unknown option: $1"
                show_help
                exit 1
                ;;
        esac
    done
}

# Show help information
show_help() {
    echo ""
    echo -e "${CYAN}Fully Automated Arcade Setup with SSL${NC}"
    echo ""
    echo -e "${YELLOW}Usage:${NC}"
    echo "  ./setup_vps.sh [--domain yourdomain.com] [--purchase-code YOUR-CODE]"
    echo ""
    echo -e "${YELLOW}Features:${NC}"
    echo "  • Auto-generates ALL passwords and credentials securely"
    echo "  • Requires domain name for SSL certificate"
    echo "  • Requires purchase code for license verification"
    echo "  • Sets up Let's Encrypt SSL with auto-renewal"
    echo "  • Nginx reverse proxy with security headers"
    echo "  • Production-ready configuration"
    echo ""
    echo -e "${YELLOW}Upload Source First:${NC}"
    echo "  Before running this script, upload the arcade source to /home/"
    echo "  Use one of these methods:"
    echo "    • rsync: rsync -avz --exclude node_modules . user@server:/home/arcade/"
    echo "    • scp: scp -r . user@server:/home/arcade/"
    echo "    • git: ssh user@server 'cd /home && git clone <repo> arcade'"
    echo ""
    echo -e "${YELLOW}Examples:${NC}"
    echo "  # Interactive (prompts for domain and purchase code):"
    echo "  cd /home/arcade && ./setup_vps.sh"
    echo ""
    echo "  # Non-interactive with command line arguments:"
    echo "  ./setup_vps.sh --domain arcade.example.com --purchase-code ABC123-DEF456"
    echo ""
    echo -e "${YELLOW}What gets auto-generated:${NC}"
    echo "  • MySQL root password (20 chars)"
    echo "  • Database user and password (16 chars)" 
    echo "  • Admin password (12 chars)"
    echo "  • Admin email (admin@yourdomain)"
    echo "  • SSL certificates via Let's Encrypt"
    echo ""
    echo -e "${YELLOW}User must provide:${NC}"
    echo "  • Domain name for SSL certificate"
    echo "  • Purchase code for license verification (alphanumeric + hyphens)"
    echo ""
}

# No validation needed - everything auto-generated

# Check if running as root
check_root() {
    if [[ $EUID -eq 0 ]]; then
        log_warning "Running as root user - proceeding with root privileges"
        # When running as root, we don't need sudo commands
        # Replace sudo commands with direct execution by creating a sudo alias
        alias sudo=''
    else
        # Check sudo access for non-root users
        if ! sudo -n true 2>/dev/null; then
            log_error "This script requires sudo privileges"
            exit 1
        fi
        log_info "Running as non-root user with sudo privileges"
    fi
}

# Update system packages
update_system() {
    log_header "UPDATING SYSTEM PACKAGES"

    log_wait "Updating package repositories"
    sudo apt update -y >/dev/null 2>&1

    log_wait "Upgrading system packages"
    sudo apt upgrade -y >/dev/null 2>&1

    log_wait "Installing essential packages"
    sudo apt install -y curl wget gnupg2 software-properties-common apt-transport-https ca-certificates git openssl >/dev/null 2>&1

    log_success "System packages updated successfully"
}

# Install MySQL
install_mysql() {
    log_header "INSTALLING MYSQL 8.0"

    log_progress "Configuring MySQL installation settings"
    # Set MySQL root password for unattended installation
    sudo debconf-set-selections <<< "mysql-server mysql-server/root_password password $MYSQL_ROOT_PASSWORD"
    sudo debconf-set-selections <<< "mysql-server mysql-server/root_password_again password $MYSQL_ROOT_PASSWORD"

    log_wait "Installing MySQL server and client"
    sudo apt install -y mysql-server mysql-client >/dev/null 2>&1

    log_progress "Starting MySQL service"
    sudo systemctl start mysql
    sudo systemctl enable mysql

    log_wait "Securing MySQL installation - please wait while we configure security settings"

    # Wait for MySQL to be ready
    sleep 5
    
    # Set root password and secure installation
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';" 2>/dev/null || {
        # If password auth fails, try without password first (fresh install)
        sudo mysql -e "ALTER USER 'root'@'localhost' IDENTIFIED WITH mysql_native_password BY '$MYSQL_ROOT_PASSWORD';"
    }
    
    # Secure the installation
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "DELETE FROM mysql.user WHERE User='';"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "DELETE FROM mysql.user WHERE User='root' AND Host NOT IN ('localhost', '127.0.0.1', '::1');"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "DROP DATABASE IF EXISTS test;"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "DELETE FROM mysql.db WHERE Db='test' OR Db='test\\_%';"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "FLUSH PRIVILEGES;"
    
    log_success "MySQL 8.0 installed and secured"
}

# Create database and user
create_database() {
    log_header "CREATING ARCADE DATABASE"
    
    # Create database
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "CREATE DATABASE IF NOT EXISTS $DB_NAME CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;"
    
    # Create user and grant privileges
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "CREATE USER IF NOT EXISTS '$DB_USERNAME'@'localhost' IDENTIFIED BY '$DB_PASSWORD';"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "GRANT ALL PRIVILEGES ON $DB_NAME.* TO '$DB_USERNAME'@'localhost';"
    mysql -u root -p"$MYSQL_ROOT_PASSWORD" -e "FLUSH PRIVILEGES;"
    
    # Test connection
    if mysql -u "$DB_USERNAME" -p"$DB_PASSWORD" -e "USE $DB_NAME; SELECT 1;" >/dev/null 2>&1; then
        log_success "Database '$DB_NAME' created successfully"
        log_success "User '$DB_USERNAME' created with full privileges"
    else
        log_error "Failed to create database or user"
        exit 1
    fi
}

# Install Node.js
install_nodejs() {
    log_header "INSTALLING NODE.JS 20 LTS"

    log_wait "Adding NodeSource repository - this may take a moment"
    curl -fsSL https://deb.nodesource.com/setup_20.x | sudo -E bash - >/dev/null 2>&1

    log_wait "Installing Node.js 20 LTS"
    sudo apt install -y nodejs >/dev/null 2>&1

    # Verify installation
    NODE_VERSION=$(node --version)
    NPM_VERSION=$(npm --version)

    log_success "Node.js installed: $NODE_VERSION"
    log_success "npm installed: $NPM_VERSION"

    log_wait "Installing PM2 process manager globally"
    sudo npm install -g pm2 >/dev/null 2>&1
    log_success "PM2 process manager installed globally"
}

# Install and configure Nginx
install_nginx() {
    log_header "INSTALLING NGINX"

    log_wait "Installing Nginx web server"
    sudo apt install -y nginx >/dev/null 2>&1

    log_progress "Starting and enabling Nginx service"
    sudo systemctl start nginx
    sudo systemctl enable nginx

    # Create Nginx configuration based on domain type
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        # Root domain with www redirect
        sudo tee "/etc/nginx/sites-available/$CANONICAL_DOMAIN" > /dev/null <<EOF
# WWW to Non-WWW Redirect
server {
    listen 80;
    server_name $WWW_DOMAIN;
    return 301 http://$CANONICAL_DOMAIN\$request_uri;
}

# Main Server Block (Canonical)
server {
    listen 80;
    server_name $CANONICAL_DOMAIN;

    # Security headers
    add_header X-Frame-Options DENY;
    add_header X-Content-Type-Options nosniff;
    add_header X-XSS-Protection "1; mode=block";

    # Client body size limit (for file uploads)
    client_max_body_size 100M;

    # Proxy settings for Node.js app
    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_cache_bypass \$http_upgrade;
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
        proxy_buffer_size 64k;
        proxy_buffers 16 32k;
        proxy_busy_buffers_size 64k;
    }

    # Static files optimization
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)\$ {
        proxy_pass http://localhost:3000;
        expires 1y;
        add_header Cache-Control "public, immutable";
        add_header Vary Accept-Encoding;
        access_log off;
    }

    # Security - block access to sensitive files
    location ~ /\.(env|git|htaccess|htpasswd) {
        deny all;
        return 404;
    }

    # Block access to backup and config files
    location ~ \.(bak|config|sql|fla|psd|ini|log|sh|inc|swp|dist)\$ {
        deny all;
        return 404;
    }

    # Block common exploit attempts
    location ~ /(wp-admin|wp-login|phpmyadmin) {
        deny all;
        return 404;
    }
}
EOF
    else
        # Subdomain - single server block
        sudo tee "/etc/nginx/sites-available/$CANONICAL_DOMAIN" > /dev/null <<EOF
server {
    listen 80;
    server_name $CANONICAL_DOMAIN;

    # Security headers
    add_header X-Frame-Options DENY;
    add_header X-Content-Type-Options nosniff;
    add_header X-XSS-Protection "1; mode=block";

    # Client body size limit (for file uploads)
    client_max_body_size 100M;

    # Proxy settings for Node.js app
    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
        proxy_cache_bypass \$http_upgrade;
        proxy_connect_timeout 60s;
        proxy_send_timeout 60s;
        proxy_read_timeout 60s;
        proxy_buffer_size 64k;
        proxy_buffers 16 32k;
        proxy_busy_buffers_size 64k;
    }

    # Static files optimization
    location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf|eot)\$ {
        proxy_pass http://localhost:3000;
        expires 1y;
        add_header Cache-Control "public, immutable";
        add_header Vary Accept-Encoding;
        access_log off;
    }

    # Security - block access to sensitive files
    location ~ /\.(env|git|htaccess|htpasswd) {
        deny all;
        return 404;
    }

    # Block access to backup and config files
    location ~ \.(bak|config|sql|fla|psd|ini|log|sh|inc|swp|dist)\$ {
        deny all;
        return 404;
    }

    # Block common exploit attempts
    location ~ /(wp-admin|wp-login|phpmyadmin) {
        deny all;
        return 404;
    }
}
EOF
    fi

    # Enable the site using canonical domain name
    sudo ln -sf "/etc/nginx/sites-available/$CANONICAL_DOMAIN" /etc/nginx/sites-enabled/
    sudo rm -f /etc/nginx/sites-enabled/default

    # Test and reload nginx
    sudo nginx -t
    sudo systemctl reload nginx

    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        log_success "Nginx installed and configured for $CANONICAL_DOMAIN with www redirect"
    else
        log_success "Nginx installed and configured for $CANONICAL_DOMAIN"
    fi
}

# Install SSL certificate with Let's Encrypt
install_ssl() {
    log_header "INSTALLING SSL CERTIFICATE"
    
    # Install Certbot
    sudo apt install -y certbot python3-certbot-nginx
    
    # Display server IP for DNS configuration (force IPv4)
    SERVER_IP=$(curl -s -4 ifconfig.me || curl -s -4 ipecho.net/plain || curl -s -4 icanhazip.com || echo "Unable to detect IP")
    log_info "Server IP Address: $SERVER_IP"
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        log_warning "IMPORTANT: Make sure your domains point to this server's IP!"
        log_info "Required A records:"
        log_info "  $CANONICAL_DOMAIN -> $SERVER_IP"
        log_info "  $WWW_DOMAIN -> $SERVER_IP"
    else
        log_warning "IMPORTANT: Make sure your domain '$CANONICAL_DOMAIN' points to this server's IP!"
        log_info "Add an A record: $CANONICAL_DOMAIN -> $SERVER_IP"
    fi
    
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${MAGENTA}                                    🔒 DNS CONFIGURATION REQUIRED 🔒${NC}"
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}⚠️  USER ACTION REQUIRED - CONFIGURE DNS BEFORE CONTINUING ⚠️${NC}"
    echo ""
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        echo -e "${YELLOW}IMPORTANT: Make sure both domains point to this server's IP!${NC}"
        echo -e "${GREEN}Required DNS Records:${NC}"
        echo -e "  ${CYAN}$CANONICAL_DOMAIN ${GREEN}→ ${CYAN}$SERVER_IP${NC}"
        echo -e "  ${CYAN}$WWW_DOMAIN ${GREEN}→ ${CYAN}$SERVER_IP${NC}"
        echo ""
        echo -e "${YELLOW}You can verify DNS propagation with:${NC}"
        echo -e "  • Command line: ${CYAN}nslookup $CANONICAL_DOMAIN && nslookup $WWW_DOMAIN${NC}"
        echo -e "  • Online tool: ${CYAN}https://dnschecker.org${NC}"
    else
        echo -e "${YELLOW}IMPORTANT: Make sure your domain '${CYAN}$CANONICAL_DOMAIN${YELLOW}' points to this server's IP!${NC}"
        echo -e "${GREEN}Required DNS Record: ${CYAN}$CANONICAL_DOMAIN ${GREEN}→ ${CYAN}$SERVER_IP${NC}"
        echo ""
        echo -e "${YELLOW}You can verify DNS propagation with:${NC}"
        echo -e "  • Command line: ${CYAN}nslookup $CANONICAL_DOMAIN${NC}"
        echo -e "  • Online tool: ${CYAN}https://dnschecker.org${NC}"
    fi
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -en "${CYAN}${BOLD}>>> Press Enter when DNS is configured and ready: ${NC}"
    read
    
    # Verify domain resolution before attempting SSL
    log_info "Verifying domain resolution..."
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        # Check both domains
        CANONICAL_OK=false
        WWW_OK=false

        if nslookup "$CANONICAL_DOMAIN" >/dev/null 2>&1; then
            RESOLVED_IP=$(nslookup "$CANONICAL_DOMAIN" | awk '/^Address: / { print $2 }' | tail -1)
            if [[ "$RESOLVED_IP" == "$SERVER_IP" ]]; then
                log_success "Domain $CANONICAL_DOMAIN resolves correctly to this server"
                CANONICAL_OK=true
            else
                log_warning "Domain $CANONICAL_DOMAIN resolves to $RESOLVED_IP but server IP is $SERVER_IP"
            fi
        else
            log_warning "Could not resolve $CANONICAL_DOMAIN"
        fi

        if nslookup "$WWW_DOMAIN" >/dev/null 2>&1; then
            RESOLVED_IP=$(nslookup "$WWW_DOMAIN" | awk '/^Address: / { print $2 }' | tail -1)
            if [[ "$RESOLVED_IP" == "$SERVER_IP" ]]; then
                log_success "Domain $WWW_DOMAIN resolves correctly to this server"
                WWW_OK=true
            else
                log_warning "Domain $WWW_DOMAIN resolves to $RESOLVED_IP but server IP is $SERVER_IP"
            fi
        else
            log_warning "Could not resolve $WWW_DOMAIN"
        fi

        if [[ "$CANONICAL_OK" == "false" ]] || [[ "$WWW_OK" == "false" ]]; then
            log_warning "SSL may fail - continuing anyway..."
        fi
    else
        # Check single domain
        if nslookup "$CANONICAL_DOMAIN" >/dev/null 2>&1; then
            RESOLVED_IP=$(nslookup "$CANONICAL_DOMAIN" | awk '/^Address: / { print $2 }' | tail -1)
            if [[ "$RESOLVED_IP" == "$SERVER_IP" ]]; then
                log_success "Domain resolves correctly to this server"
            else
                log_warning "Domain resolves to $RESOLVED_IP but server IP is $SERVER_IP"
                log_warning "SSL may fail - continuing anyway..."
            fi
        else
            log_warning "Could not resolve domain - SSL may fail"
        fi
    fi
    
    # Get SSL certificate based on domain type
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        # Root domain - get certificate for both domains
        log_info "Obtaining SSL certificate for $CANONICAL_DOMAIN and $WWW_DOMAIN..."
        if sudo certbot --nginx -d "$CANONICAL_DOMAIN" -d "$WWW_DOMAIN" --non-interactive --agree-tos --email "$ADMIN_EMAIL" --redirect; then
            log_success "SSL certificate installed for both $CANONICAL_DOMAIN and $WWW_DOMAIN"
            log_success "HTTPS redirect enabled"
        else
            log_error "Failed to obtain SSL certificate for root domain"
            return 1
        fi
    else
        # Subdomain - single certificate
        log_info "Obtaining SSL certificate for $CANONICAL_DOMAIN..."
        if sudo certbot --nginx -d "$CANONICAL_DOMAIN" --non-interactive --agree-tos --email "$ADMIN_EMAIL" --redirect; then
            log_success "SSL certificate installed for $CANONICAL_DOMAIN"
            log_success "HTTPS redirect enabled"
        else
            log_error "Failed to obtain SSL certificate for subdomain"
            return 1
        fi
    fi

    # Set up auto-renewal
    sudo systemctl enable certbot.timer
    sudo systemctl start certbot.timer

    # Test auto-renewal
    if sudo certbot renew --dry-run >/dev/null 2>&1; then
        log_success "SSL auto-renewal configured and tested"
    else
        log_warning "SSL installed but auto-renewal test failed"
    fi

    log_success "SSL setup completed successfully!"
}

# Configure firewall
configure_firewall() {
    log_header "CONFIGURING FIREWALL"
    
    # Install and enable UFW
    sudo apt install -y ufw
    
    # Configure rules for Nginx proxy setup
    sudo ufw default deny incoming
    sudo ufw default allow outgoing
    sudo ufw allow ssh
    sudo ufw allow 'Nginx Full'
    sudo ufw allow 3306     # MySQL (remove if not needed externally)
    
    # Enable firewall
    sudo ufw --force enable
    
    log_success "Firewall configured"
}

# Setup arcade application from uploaded source
setup_application() {
    log_header "SETTING UP ARCADE APPLICATION"
    
    # Check if we're already in the correct directory
    if [[ -f "app.js" && -f "package.json" ]]; then
        log_info "Found arcade source files in current directory"
        ARCADE_DIR=$(pwd)
    else
        # Look for arcade files in common locations
        ARCADE_PATHS=(
            "/home/arcade"
            "/home/titansystems-arcade" 
            "/home/$(whoami)/arcade"
            "/home/$(whoami)/titansystems-arcade"
            "/home/*/arcade"
            "/home/*/titansystems-arcade"
        )
        
        ARCADE_DIR=""
        for path in "${ARCADE_PATHS[@]}"; do
            # Handle wildcard paths
            for dir in $path; do
                if [[ -d "$dir" && -f "$dir/app.js" && -f "$dir/package.json" ]]; then
                    ARCADE_DIR="$dir"
                    break 2
                fi
            done
        done
        
        if [[ -z "$ARCADE_DIR" ]]; then
            log_error "Could not find arcade source files!"
            log_info "Please ensure the arcade source is uploaded to one of these locations:"
            log_info "  - /home/arcade/"
            log_info "  - /home/titansystems-arcade/"
            log_info "  - /home/\$USER/arcade/"
            log_info "  - /home/\$USER/titansystems-arcade/"
            exit 1
        fi
        
        log_info "Found arcade source at: $ARCADE_DIR"
        cd "$ARCADE_DIR"
    fi
    
    # Verify we have the required files
    if [[ ! -f "app.js" || ! -f "package.json" || ! -f "scripts/deploy_vps.js" ]]; then
        log_error "Missing required files in arcade directory!"
        log_info "Required files: app.js, package.json, scripts/deploy_vps.js"
        exit 1
    fi
    
    log_info "Setting proper permissions..."
    # Ensure the current user owns the files
    CURRENT_USER=$(whoami)
    sudo chown -R "$CURRENT_USER:$CURRENT_USER" .
    
    # Make scripts executable
    chmod +x scripts/*.js 2>/dev/null || true
    chmod +x scripts/*.sh 2>/dev/null || true
    
    # Install npm dependencies
    log_wait "Installing npm dependencies - this may take several minutes"
    npm install >/dev/null 2>&1

    # Build assets
    log_wait "Building application assets - please wait while we compile CSS and JavaScript"
    npm run build >/dev/null 2>&1 || {
        log_warning "Build command failed, but continuing..."
    }
    
    log_success "Application setup completed"
    log_info "Working directory: $(pwd)"
}

# Deploy arcade platform
deploy_arcade() {
    log_header "DEPLOYING ARCADE PLATFORM"
    
    # Ensure MySQL is ready before running deployment
    log_info "Verifying MySQL readiness..."
    local retry_count=0
    local max_retries=30
    
    while [[ $retry_count -lt $max_retries ]]; do
        if mysql -u "$DB_USERNAME" -p"$DB_PASSWORD" -h "$DB_HOST" -D "$DB_NAME" -e "SELECT 1;" >/dev/null 2>&1; then
            log_success "MySQL is ready and accessible"
            break
        else
            log_warning "MySQL not ready yet, waiting... (attempt $((retry_count + 1))/$max_retries)"
            sleep 2
            retry_count=$((retry_count + 1))
        fi
    done
    
    if [[ $retry_count -eq $max_retries ]]; then
        log_error "MySQL failed to become ready within 60 seconds"
        log_error "Please check MySQL service status: sudo systemctl status mysql"
        exit 1
    fi
    
    log_wait "Running automated deployment - this is the longest step, please be patient"

    # Verify deploy script exists
    if [[ ! -f "scripts/deploy_vps.js" ]]; then
        log_error "Deploy script not found: scripts/deploy_vps.js"
        log_info "Please ensure the arcade source code is complete"
        exit 1
    fi

    # Log deployment parameters (without sensitive info)
    log_progress "Deployment parameters configured:"
    log_info "  Site: $SITE_NAME"
    log_info "  URL: $SITE_URL"
    log_info "  Port: $SERVER_PORT"
    log_info "  Admin: $ADMIN_USERNAME"
    log_info "  Database: $DB_NAME@$DB_HOST"
    log_info "  Purchase Code: ${PURCHASE_CODE:0:10}..."

    # Run the automated deployment script with timeout and enhanced logging
    echo -e "${YELLOW}[WAIT]${NC} Executing deployment script - ${CYAN}This may take up to 10 minutes, please do not interrupt...${NC}"
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [INFO] Starting Node.js deployment script..." >> "$INSTALL_LOG"

    # Create a temporary file to capture output
    TEMP_OUTPUT=$(mktemp)

    # Run deployment with timeout (10 minutes = 600 seconds)
    if timeout 600 node scripts/deploy_vps.js \
        --site-name "$SITE_NAME" \
        --site-description "$SITE_DESCRIPTION" \
        --purchase-code "$PURCHASE_CODE" \
        --site-url "$SITE_URL" \
        --port "$SERVER_PORT" \
        --admin-username "$ADMIN_USERNAME" \
        --admin-first-name "$ADMIN_FIRST_NAME" \
        --admin-last-name "$ADMIN_LAST_NAME" \
        --admin-email "$ADMIN_EMAIL" \
        --admin-password "$ADMIN_PASSWORD" \
        --db-host "$DB_HOST" \
        --db-name "$DB_NAME" \
        --db-username "$DB_USERNAME" \
        --db-password "$DB_PASSWORD" 2>&1 | tee "$TEMP_OUTPUT"; then

        # Success case
        log_success "Arcade platform deployed successfully!"
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [SUCCESS] Node.js deployment completed successfully" >> "$INSTALL_LOG"

        # Append deployment output to log
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [INFO] Deployment script output:" >> "$INSTALL_LOG"
        sed 's/^/  /' "$TEMP_OUTPUT" >> "$INSTALL_LOG"

    else
        DEPLOY_EXIT_CODE=$?

        # Check if it was a timeout
        if [[ $DEPLOY_EXIT_CODE -eq 124 ]]; then
            log_error "Deployment script timed out after 10 minutes!"
            echo "[$(date '+%Y-%m-%d %H:%M:%S')] [ERROR] Deployment timed out after 600 seconds" >> "$INSTALL_LOG"
        else
            log_error "Arcade deployment failed with exit code: $DEPLOY_EXIT_CODE"
            echo "[$(date '+%Y-%m-%d %H:%M:%S')] [ERROR] Deployment failed with exit code: $DEPLOY_EXIT_CODE" >> "$INSTALL_LOG"
        fi

        # Append failure output to log
        echo "[$(date '+%Y-%m-%d %H:%M:%S')] [ERROR] Deployment script output:" >> "$INSTALL_LOG"
        sed 's/^/  /' "$TEMP_OUTPUT" >> "$INSTALL_LOG"

        log_error "Common causes:"
        log_info "  - Database connection timeout (check MySQL service)"
        log_info "  - Missing npm dependencies (check package.json)"
        log_info "  - Migration execution timeout (check migration files)"
        log_info "  - Invalid credentials or configuration"
        log_info "  - Network connectivity issues"
        log_info ""
        log_info "Check the complete log in: $INSTALL_LOG"

        # Clean up and exit
        rm -f "$TEMP_OUTPUT"
        exit 1
    fi

    # Clean up temporary file
    rm -f "$TEMP_OUTPUT"
}

# Setup PM2 for the arcade app
create_service() {
    log_header "SETTING UP PM2 PROCESS MANAGER"

    # Run PM2 commands as requested
    log_progress "Checking PM2 status"
    pm2

    log_wait "Starting Arcade application with PM2 - please wait"
    pm2 start app.js --name "Arcade"

    log_progress "Saving PM2 configuration"
    pm2 save

    log_success "PM2 setup completed successfully"
}

# Verify PM2 service is running
start_service() {
    log_header "VERIFYING PM2 SERVICE STATUS"

    # Wait a moment for PM2 to fully initialize
    sleep 3

    # Check PM2 status
    log_info "Checking PM2 application status..."
    if pm2 list | grep -q "Arcade.*online"; then
        log_success "Arcade application is running successfully with PM2!"

        # Show PM2 status
        echo ""
        pm2 list
        echo ""
        pm2 show Arcade
        echo ""
    else
        log_error "Failed to start arcade application with PM2!"
        log_info "Check PM2 logs with: pm2 logs Arcade"
        log_info "Restart with: pm2 restart Arcade"
        exit 1
    fi
}

# Save credentials (removed file creation)
save_credentials() {
    log_header "PREPARING CREDENTIALS FOR DISPLAY"
    
    log_info "All credentials ready for final display"
}

# Display final information
show_final_info() {
    log_highlight "🚀 AUTOMATED SETUP WITH SSL COMPLETED! 🚀"
    
    echo -e "${GREEN}Complete arcade platform with SSL is now live!${NC}\n"
    
    echo -e "${YELLOW}🌐 ACCESS YOUR PLATFORM:${NC}"
    if [[ "$NEEDS_WWW_REDIRECT" == "true" ]]; then
        echo -e "  Primary URL: ${CYAN}$SITE_URL${NC}"
        echo -e "  Alternate: ${CYAN}https://$WWW_DOMAIN${NC} ${GREEN}(redirects to primary)${NC}\n"
    else
        echo -e "  Website: ${CYAN}$SITE_URL${NC}\n"
    fi
    
    echo -e "${YELLOW}🔐 AUTO-GENERATED CREDENTIALS:${NC}"
    echo -e "${CYAN}Admin Login:${NC}"
    echo -e "  Username: ${ADMIN_USERNAME}"
    echo -e "  Email: ${ADMIN_EMAIL}"
    echo -e "  Password: ${ADMIN_PASSWORD}\n"
    
    echo -e "${CYAN}Database:${NC}"
    echo -e "  Database: ${DB_NAME}"
    echo -e "  Username: ${DB_USERNAME}"
    echo -e "  Password: ${DB_PASSWORD}\n"
    
    echo -e "${YELLOW}🔒 SSL CERTIFICATE:${NC}"
    echo -e "  Status: ${GREEN}Active${NC}"
    echo -e "  Auto-renewal: ${GREEN}Enabled${NC}"
    echo -e "  Check renewal: ${CYAN}sudo certbot certificates${NC}\n"
    
    echo -e "${YELLOW}🛠️  PM2 PROCESS MANAGEMENT:${NC}"
    echo -e "  Status: ${CYAN}pm2 status${NC}"
    echo -e "  Logs: ${CYAN}pm2 logs Arcade${NC}"
    echo -e "  Restart: ${CYAN}pm2 restart Arcade${NC}"
    echo -e "  Monitor: ${CYAN}pm2 monit${NC}"
    echo -e "  Details: ${CYAN}pm2 show Arcade${NC}\n"
    
    echo -e "${YELLOW}📝 IMPORTANT:${NC}"
    echo -e "  ${RED}Please save these credentials now!${NC}\n"
    
    echo -e "${GREEN}Your secure arcade platform is now ready to use!${NC}"
}

# Main execution
main() {
    # Initialize logging first
    setup_logging "$@"

    log_highlight "TITAN SYSTEMS ARCADE - FULL AUTOMATION"

    parse_arguments "$@"
    check_root
    
    # Generate all credentials automatically
    generate_credentials
    
    # Get domain if not provided
    if [[ -z "$DOMAIN" ]]; then
        get_domain
    else
        # Validate domain if provided via command line
        if ! validate_and_process_domain "$DOMAIN"; then
            log_error "Invalid domain provided via command line"
            exit 1
        fi
        # Update URLs based on canonical domain
        SITE_URL="https://${CANONICAL_DOMAIN}"
        ADMIN_EMAIL="admin@${CANONICAL_DOMAIN}"
    fi
    
    # Get purchase code if not provided
    if [[ -z "$PURCHASE_CODE" ]]; then
        get_purchase_code
    else
        # Validate purchase code if provided via command line
        if [[ ! "$PURCHASE_CODE" =~ ^[a-zA-Z0-9-]+$ ]]; then
            log_error "Invalid purchase code format provided via command line"
            log_info "Purchase code must contain only letters, numbers, and hyphens"
            exit 1
        fi
        log_info "Purchase code accepted: ${PURCHASE_CODE:0:10}..."
    fi
    
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -e "${MAGENTA}                                  🚀 READY TO BEGIN INSTALLATION 🚀${NC}"
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo ""
    echo -e "${CYAN}${BOLD}⚠️  FINAL CONFIRMATION - REVIEW SETTINGS BEFORE PROCEEDING ⚠️${NC}"
    echo ""
    echo -e "${YELLOW}Setup Preview:${NC}"
    echo -e "${GREEN}  ✓ Domain: ${CYAN}$CANONICAL_DOMAIN${NC}"
    echo -e "${GREEN}  ✓ Site URL: ${CYAN}$SITE_URL${NC}"
    echo -e "${GREEN}  ✓ Admin Email: ${CYAN}$ADMIN_EMAIL${NC}"
    echo -e "${GREEN}  ✓ Purchase Code: ${CYAN}${PURCHASE_CODE:0:10}...${NC}"
    echo -e "${GREEN}  ✓ SSL: ${CYAN}Let's Encrypt (auto-renewal enabled)${NC}"
    echo -e "${GREEN}  ✓ Passwords: ${CYAN}Auto-generated securely${NC}"
    echo ""
    echo -e "${YELLOW}This will install and configure:${NC}"
    echo -e "${CYAN}  • MySQL 8.0 Database Server${NC}"
    echo -e "${CYAN}  • Node.js 20 LTS Runtime${NC}"
    echo -e "${CYAN}  • Nginx Web Server with SSL${NC}"
    echo -e "${CYAN}  • Complete Arcade Platform${NC}"
    echo -e "${CYAN}  • System Service Configuration${NC}"
    echo ""
    echo -e "${MAGENTA}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${NC}"
    echo -en "${CYAN}${BOLD}>>> Continue with automated setup? (y/N): ${NC}"
    read -n 1 -r REPLY
    echo ""

    if [[ ! $REPLY =~ ^[Yy]$ ]]; then
        echo ""
        log_warning "Setup cancelled by user"
        log_info "You can run this script again anytime to start the installation"
        exit 0
    fi

    echo ""
    log_success "Installation confirmed! Starting automated setup..."
    echo ""
    
    update_system
    install_mysql
    create_database
    install_nodejs
    install_nginx
    install_ssl
    configure_firewall
    setup_application
    deploy_arcade
    create_service
    start_service
    save_credentials

    show_final_info

    log_success "Full automation setup completed!"

    # Final log entry and clean exit
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] [SUCCESS] Setup completed successfully - exiting" >> "$INSTALL_LOG"
    exit 0
}

# Run main function
main "$@"