/**
 * Migration: add mobile compatibility to games
 * 
 * Adds is_mobile_compatible column to games table to control whether
 * games are shown to mobile device users.
 * 
 * IMPORTANT: Follow database guidelines:
 * - Use only simple column types: tinyint, int, bigint, float, tinytext, text, longtext, datetime, date, time
 * - NO forbidden types: varchar, char, timestamp, enum, set, json, decimal
 * - TEXT columns cannot have DEFAULT values
 * - Always use utf8mb4_unicode_ci collation
 */

import {
    tableExists,
    columnExists,
    addColumnIfNotExists
} from '../src/utils/migration.js';

/**
 * Run the migration (forward)
 * @param {object} connection - MySQL connection object
 */
export async function up(connection) {
    // Ensure games table exists
    const gamesTableExists = await tableExists(connection, 'games');
    if (!gamesTableExists) {
        throw new Error('Games table must exist before adding mobile compatibility column');
    }
    
    // Add mobile compatibility column to games table
    const columnResult = await addColumnIfNotExists(
        connection,
        'games',
        'is_mobile_compatible',
        'is_mobile_compatible tinyint(1) DEFAULT 0'
    );
    
    console.log(`Migration: ${columnResult.message}`);
}

/**
 * Reverse the migration (rollback)
 * @param {object} connection - MySQL connection object  
 */
export async function down(connection) {
    // Check if column exists before dropping
    const exists = await columnExists(connection, 'games', 'is_mobile_compatible');
    if (exists) {
        // Drop column
        await connection.execute('ALTER TABLE games DROP COLUMN is_mobile_compatible');
        console.log('Migration rollback: Column is_mobile_compatible dropped from games table');
    } else {
        console.log('Migration rollback: Column is_mobile_compatible does not exist, skipping drop');
    }
}

/**
 * Optional: SQL for emergency rollback (stored in database)
 */
export const rollback = `
ALTER TABLE games DROP COLUMN IF EXISTS is_mobile_compatible;
`;
