/**
 * Migration: Add TIC-80 Advertisement Placements
 *
 * Adds dedicated advertisement placement types for TIC-80 games:
 * - TIC-80 Game Pre-Roll (tic80-preroll)
 * - TIC-80 Game Interstitial (tic80-interstitial)
 *
 * These placements allow for TIC-80-specific ad targeting and management.
 */

/**
 * Run the migration (up)
 * @param {object} connection - MySQL connection object
 */
export async function up(connection) {
    console.log('Adding TIC-80 advertisement placements...');

    // Add TIC-80 specific advertisement placements
    await connection.execute(`
        INSERT INTO \`ad_placements\` (\`name\`, \`slug\`, \`description\`, \`width\`, \`height\`, \`placement_type\`, \`is_active\`) VALUES
        ('TIC-80 Game Pre-Roll', 'tic80-preroll', 'Pre-roll ad shown before TIC-80 games start', 400, 300, 'tic80', 1),
        ('TIC-80 Game Interstitial', 'tic80-interstitial', 'Interstitial ad during TIC-80 gameplay', 400, 300, 'tic80', 1);
    `);

    console.log('✅ TIC-80 advertisement placements added successfully');
}

/**
 * Reverse the migration (rollback)
 * @param {object} connection - MySQL connection object
 */
export async function down(connection) {
    console.log('Removing TIC-80 advertisement placements...');

    // Remove TIC-80 ads first (foreign key constraint)
    await connection.execute(`
        DELETE FROM \`ads\` WHERE \`placement_id\` IN (
            SELECT \`id\` FROM \`ad_placements\` WHERE \`slug\` IN ('tic80-preroll', 'tic80-interstitial')
        );
    `);

    // Remove TIC-80 placements by slug (more reliable than hard-coded IDs)
    await connection.execute(`
        DELETE FROM \`ad_placements\` WHERE \`slug\` IN ('tic80-preroll', 'tic80-interstitial');
    `);

    console.log('✅ TIC-80 advertisement placements removed successfully');
}

/**
 * SQL for emergency rollback (stored in database)
 */
export const rollback = `
    DELETE FROM ads WHERE placement_id IN (
        SELECT id FROM ad_placements WHERE slug IN ('tic80-preroll', 'tic80-interstitial')
    );
    DELETE FROM ad_placements WHERE slug IN ('tic80-preroll', 'tic80-interstitial');
`;