# Example Plugin

A simple demonstration of the Arcade Plugin System featuring basic EJS template rendering and sample games JSON response.

## Overview

This plugin showcases the minimal implementation of a plugin with two basic actions. It serves as a starting reference for developers building their own plugins.

## Features

- **Simple Hello World Template**: Basic EJS template rendering
- **Sample Games JSON API**: Returns structured JSON data
- **Clean Architecture**: Minimal, well-documented code structure
- **Self-Contained Design**: No external dependencies

## Plugin Structure

```
plugins/example-plugin/
├── info.json            # Plugin metadata and configuration
├── plugin.js            # Plugin logic with action handlers
├── views/               # EJS templates
│   └── default.ejs      # Simple Hello World page
├── assets/              # Static files
│   ├── css/style.css    # Plugin-specific styles
│   └── js/script.js     # Plugin-specific JavaScript
└── README.md            # This documentation
```

## Available Actions

### 1. Default Action
- **URL**: `/plugin?name=example-plugin&action=default` or `/plugin?name=example-plugin`
- **Method**: GET
- **Purpose**: Renders a simple "Hello World" page
- **Response**: Basic HTML page

### 2. Games Action
- **URL**: `/plugin?name=example-plugin&action=games`
- **Method**: GET 
- **Purpose**: Returns a JSON list of sample games data
- **Response**: JSON with games array and metadata

**Sample Games JSON Response**:
```json
{
  "success": true,
  "message": "🎮 Games list retrieved successfully!",
  "data": {
    "games": [
      {
        "id": 1,
        "title": "Super Mario Bros",
        "system": "NES",
        "category": "Platform",
        "description": "Classic side-scrolling platformer adventure",
        "thumbnail": "/uploads/games/mario-bros-thumb.jpg",
        "file_path": "/uploads/roms/mario-bros.nes",
        "file_size": 40960,
        "plays": 15420,
        "rating": 4.8,
        "release_year": 1985,
        "developer": "Nintendo",
        "is_featured": true,
        "date_added": "2024-01-15T10:30:00Z"
      }
    ],
    "total_count": 5,
    "generated_at": "2024-01-28T10:30:00.000Z"
  }
}
```

## Key Implementation Notes

### Self-Contained Templates
Plugin templates **MUST** be completely self-contained and **CANNOT** use site template partials like `header.ejs` or `footer.ejs`. Each template must include complete HTML document structure.

### Plugin Metadata (info.json)
All plugin metadata is stored in `info.json`:
- Plugin name, title, and description
- Available actions array (`["default", "games"]`)
- Features and capabilities
- Version and author information

### Action Handlers (plugin.js)
Each action is defined as a function in the exported object:
```javascript
export default {
  "default": async (req, res, request) => {
    // Simple Hello World template
    const pageData = {
      title: "🚀 Example Plugin Demo",
      message: "Hello World! This is a simple plugin demonstration.",
      timestamp: new Date().toISOString()
    };
    res.render("../plugins/example-plugin/views/default", pageData);
  },
  
  "games": async (req, res, request) => {
    // Return sample games JSON
    response(res, 200, false, {
      success: true,
      message: "🎮 Games list retrieved successfully!",
      data: { games: sampleGames, total_count: 5, generated_at: new Date().toISOString() }
    });
  }
};
```

## Development Reference

This plugin demonstrates:

1. **Basic Template Rendering**: How to render simple EJS templates
2. **JSON API Response**: How to return structured data
3. **Error Handling**: Proper try-catch and error redirection
4. **File Structure**: Minimal plugin organization
5. **Self-Contained Design**: Templates independent of site structure

## Usage Examples

**View Hello World Page**:
```
GET /plugin?name=example-plugin
```

**Get Games JSON Data**:
```
GET /plugin?name=example-plugin&action=games
```

**Plugin Assets**:
```
GET /plugin/assets/css/style.css?name=example-plugin
GET /plugin/assets/js/script.js?name=example-plugin
```

## Version

Current version: **2.0.0**

## Author

**Arcade Plugin System**
- GitHub: https://github.com/your-repo/arcade