/**
 * 🚀 Example Plugin - Simple Plugin System Demonstration
 * 
 * This plugin showcases basic capabilities of the Arcade Plugin System:
 * - Simple Hello World EJS template rendering
 * - Sample games JSON API response
 * - Clean, minimal plugin structure
 * 
 * Use this as a reference when building your own plugins!
 */

// Import core utilities (relative paths from plugin directory)
import response from '../../src/utils/response.js';

/**
 * Plugin Actions Object
 * Each key represents an action that can be called via: /plugin?name=example-plugin&action=<key>
 */
export default {
  /**
   * 🏠 DEFAULT ACTION
   * 
   * URL: /plugin?name=example-plugin&action=default (or just /plugin?name=example-plugin)
   * Purpose: Simple Hello World page
   * Method: GET
   */
  "default": async (req, res, request) => {
    try {
      // Render simple Hello World template
      res.render("../plugins/example-plugin/views/default");
      
    } catch (error) {
      console.error('[Example Plugin] Default action error:', error);
      res.redirect('/errors/500');
    }
  },

  /**
   * 🎮 GAMES ACTION
   * 
   * URL: /plugin?name=example-plugin&action=games
   * Purpose: Returns JSON list of sample games
   * Method: GET
   */
  "games": async (req, res, request) => {
    try {
      // Sample games JSON response (what would come from database)
      const sampleGames = [
        {
          id: 1,
          title: "Super Mario Bros",
          system: "NES",
          category: "Platform",
          description: "Classic side-scrolling platformer adventure",
          thumbnail: "/uploads/games/mario-bros-thumb.jpg",
          file_path: "/uploads/roms/mario-bros.nes",
          file_size: 40960,
          plays: 15420,
          rating: 4.8,
          release_year: 1985,
          developer: "Nintendo",
          is_featured: true,
          date_added: "2024-01-15T10:30:00Z"
        },
        {
          id: 2,
          title: "Sonic the Hedgehog",
          system: "Genesis",
          category: "Platform", 
          description: "Fast-paced blue hedgehog adventure",
          thumbnail: "/uploads/games/sonic-thumb.jpg",
          file_path: "/uploads/roms/sonic.md",
          file_size: 524288,
          plays: 12350,
          rating: 4.6,
          release_year: 1991,
          developer: "Sega",
          is_featured: true,
          date_added: "2024-01-12T14:45:00Z"
        },
        {
          id: 3,
          title: "Street Fighter II",
          system: "SNES",
          category: "Fighting",
          description: "Legendary fighting game with iconic characters", 
          thumbnail: "/uploads/games/sf2-thumb.jpg",
          file_path: "/uploads/roms/sf2.smc",
          file_size: 3145728,
          plays: 8900,
          rating: 4.9,
          release_year: 1992,
          developer: "Capcom",
          is_featured: false,
          date_added: "2024-01-10T09:15:00Z"
        },
        {
          id: 4,
          title: "The Legend of Zelda",
          system: "NES",
          category: "Adventure",
          description: "Epic fantasy adventure",
          thumbnail: "/uploads/games/zelda-thumb.jpg",
          file_path: "/uploads/roms/zelda.nes",
          file_size: 131072,
          plays: 11200,
          rating: 4.7,
          release_year: 1986,
          developer: "Nintendo",
          is_featured: true,
          date_added: "2024-01-08T16:20:00Z"
        },
        {
          id: 5,
          title: "Pac-Man",
          system: "Arcade",
          category: "Arcade",
          description: "Classic dot-eating arcade game", 
          thumbnail: "/uploads/games/pacman-thumb.jpg", 
          file_path: "/uploads/roms/pacman.zip",
          file_size: 24576,
          plays: 9800,
          rating: 4.5,
          release_year: 1980,
          developer: "Namco",
          is_featured: false,
          date_added: "2024-01-05T11:10:00Z"
        }
      ];

      // Return JSON response
      response(res, 200, "🎮 Games list retrieved successfully!", {
        games: sampleGames,
        total_count: sampleGames.length,
        generated_at: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('[Example Plugin] Games action error:', error);
      response(res, 500, "Internal server error during games retrieval");
    }
  }
};