#!/usr/bin/env node

import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { execSync } from 'child_process';
import archiver from 'archiver';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const rootDir = path.resolve(__dirname, '..');

// Colors for beautiful CLI output
const colors = {
    reset: '\x1b[0m',
    bright: '\x1b[1m',
    dim: '\x1b[2m',
    red: '\x1b[31m',
    green: '\x1b[32m',
    yellow: '\x1b[33m',
    blue: '\x1b[34m',
    magenta: '\x1b[35m',
    cyan: '\x1b[36m',
    white: '\x1b[37m'
};

// Styled console functions
const log = {
    success: (msg) => console.log(`${colors.green}✓${colors.reset} ${msg}`),
    error: (msg) => console.log(`${colors.red}✗${colors.reset} ${msg}`),
    warning: (msg) => console.log(`${colors.yellow}⚠${colors.reset} ${msg}`),
    info: (msg) => console.log(`${colors.blue}ℹ${colors.reset} ${msg}`),
    step: (msg) => console.log(`${colors.cyan}→${colors.reset} ${msg}`),
    header: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.cyan}═══════════════════════════════════════════════════════════════════════════════════${colors.reset}`);
        console.log(`${colors.bright}${colors.bgBlue}  ${msg}  ${colors.reset}`);
        console.log(`${colors.bright}${colors.cyan}═══════════════════════════════════════════════════════════════════════════════════${colors.reset}`);
        console.log('');
    },
    subheader: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.yellow}${msg}${colors.reset}`);
        console.log(`${colors.bright}${colors.blue}${'─'.repeat(msg.length)}${colors.reset}`);
        console.log('');
    },
    highlight: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.magenta}${'═'.repeat(80)}${colors.reset}`);
        console.log(`${colors.bright}${colors.bgGreen}  ${msg}  ${colors.reset}`);
        console.log(`${colors.bright}${colors.magenta}${'═'.repeat(80)}${colors.reset}`);
        console.log('');
    }
};

// Progress indicator
function showProgress(message, duration = 1000) {
    return new Promise((resolve) => {
        const frames = ['⠋', '⠙', '⠹', '⠸', '⠼', '⠴', '⠦', '⠧', '⠇', '⠏'];
        let i = 0;
        const interval = setInterval(() => {
            process.stdout.write(`\r${colors.cyan}${frames[i]}${colors.reset} ${message}...`);
            i = (i + 1) % frames.length;
        }, 100);

        setTimeout(() => {
            clearInterval(interval);
            process.stdout.write(`\r${colors.green}✓${colors.reset} ${message}\n`);
            resolve();
        }, duration);
    });
}

// Welcome banner
function showWelcomeBanner() {
    console.clear();
    console.log('');
    console.log(`${colors.bright}${colors.cyan}                  🚀 BUILD SYSTEM 🚀${colors.reset}`);
    console.log(`${colors.bright}${colors.yellow}              ✨ Production Build Generator ✨${colors.reset}`);
    console.log('');
    console.log(`${colors.dim}   Building production-ready distribution package${colors.reset}`);
    console.log(`${colors.dim}   This will create a optimized dist folder for deployment${colors.reset}`);
    console.log('');
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Asset optimization and minification${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Production-ready configuration${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Dependency management${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Clean distribution package${colors.reset}`);
    console.log('');
}

// Clean dist directory
async function cleanDistDirectory() {
    log.header('CLEANING DIST DIRECTORY');
    
    const distPath = path.join(rootDir, 'dist');
    
    try {
        if (fs.existsSync(distPath)) {
            await showProgress('Removing existing dist directory');
            fs.rmSync(distPath, { recursive: true, force: true });
            log.success('Dist directory cleaned successfully!');
        } else {
            log.info('No existing dist directory found');
        }
    } catch (error) {
        log.error('Failed to clean dist directory:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create dist directory structure
async function createDistStructure() {
    log.header('CREATING DIST STRUCTURE');
    
    const distPath = path.join(rootDir, 'dist');
    const directories = [
        'dist',
        'dist/src',
        'dist/src/controllers',
        'dist/src/middlewares',
        'dist/src/models',
        'dist/src/routers',
        'dist/src/utils',
        'dist/views',
        'dist/views/default',
        'dist/views/default/pages',
        'dist/views/default/modules',
        'dist/views/dashboard',
        'dist/views/dashboard/pages',
        'dist/views/dashboard/modules',
        'dist/views/dashboard/widgets',
        'dist/views/dashboard/widgets/modals',
        'dist/views/chatroom',
        'dist/views/chatroom/pages',
        'dist/views/mail',
        'dist/public',
        'dist/public/assets',
        'dist/public/assets/css',
        'dist/public/assets/js',
        'dist/public/assets/js/libs',
        'dist/public/assets/images',
        'dist/scripts',
        'dist/migrations',
        'dist/languages',
        'dist/plugins',
        'dist/uploads',
        'dist/uploads/avatars',
        'dist/uploads/categories',
        'dist/uploads/favicons',
        'dist/uploads/games',
        'dist/uploads/images',
        'dist/uploads/logos'
    ];
    
    try {
        await showProgress('Creating directory structure');
        
        for (const dir of directories) {
            const fullPath = path.join(rootDir, dir);
            if (!fs.existsSync(fullPath)) {
                fs.mkdirSync(fullPath, { recursive: true });
            }
        }
        
        log.success('Directory structure created successfully!');
    } catch (error) {
        log.error('Failed to create directory structure:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Copy essential files
async function copyEssentialFiles() {
    log.header('COPYING ESSENTIAL FILES');
    
    const filesToCopy = [
        { src: 'app.js', dest: 'dist/app.js' },
        { src: 'package.json', dest: 'dist/package.json' },
        { src: 'package-lock.json', dest: 'dist/package-lock.json' },
        { src: 'tailwind.config.js', dest: 'dist/tailwind.config.js' },
        { src: 'postcss.config.cjs', dest: 'dist/postcss.config.cjs' },
        { src: 'db_schema.sql', dest: 'dist/db_schema.sql' },
        { src: 'autosetup.sh', dest: 'dist/autosetup.sh' }
    ];
    
    try {
        await showProgress('Copying essential files');
        
        for (const file of filesToCopy) {
            const srcPath = path.join(rootDir, file.src);
            const destPath = path.join(rootDir, file.dest);
            
            if (fs.existsSync(srcPath)) {
                fs.copyFileSync(srcPath, destPath);
                log.info(`Copied: ${file.src}`);
            } else {
                log.warning(`File not found: ${file.src}`);
            }
        }
        
        log.success('Essential files copied successfully!');
    } catch (error) {
        log.error('Failed to copy essential files:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Copy source code
async function copySourceCode() {
    log.header('COPYING SOURCE CODE');
    
    const sourceDirs = [
        'src',
        'views',
        'scripts',
        'migrations',
        'languages',
        'plugins'
    ];
    
    try {
        await showProgress('Copying source code');
        
        for (const dir of sourceDirs) {
            const srcPath = path.join(rootDir, dir);
            const destPath = path.join(rootDir, 'dist', dir);
            
            if (fs.existsSync(srcPath)) {
                await copyDirectory(srcPath, destPath);
                log.info(`Copied directory: ${dir}`);
            } else {
                log.warning(`Directory not found: ${dir}`);
            }
        }
        
        log.success('Source code copied successfully!');
    } catch (error) {
        log.error('Failed to copy source code:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Copy public assets
async function copyPublicAssets() {
    log.header('COPYING PUBLIC ASSETS');
    
    try {
        await showProgress('Copying public assets');
        
        const publicSrcPath = path.join(rootDir, 'public');
        const publicDestPath = path.join(rootDir, 'dist', 'public');
        
        if (fs.existsSync(publicSrcPath)) {
            await copyDirectory(publicSrcPath, publicDestPath);
            log.success('Public assets copied successfully!');
        } else {
            log.warning('Public directory not found');
        }
    } catch (error) {
        log.error('Failed to copy public assets:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Copy uploads directory structure
async function copyUploadsStructure() {
    log.header('COPYING UPLOADS STRUCTURE');
    
    try {
        await showProgress('Creating uploads directory structure');
        
        const uploadsSrcPath = path.join(rootDir, 'uploads');
        const uploadsDestPath = path.join(rootDir, 'dist', 'uploads');
        
        if (fs.existsSync(uploadsSrcPath)) {
            await copyDirectory(uploadsSrcPath, uploadsDestPath);
            log.success('Uploads structure copied successfully!');
        } else {
            // Create empty uploads directories
            const uploadDirs = ['avatars', 'categories', 'favicons', 'games', 'images', 'logos'];
            for (const dir of uploadDirs) {
                const dirPath = path.join(uploadsDestPath, dir);
                if (!fs.existsSync(dirPath)) {
                    fs.mkdirSync(dirPath, { recursive: true });
                }
            }
            log.success('Empty uploads structure created successfully!');
        }
    } catch (error) {
        log.error('Failed to copy uploads structure:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create production package.json
async function createProductionPackageJson() {
    log.header('CREATING PRODUCTION PACKAGE.JSON');
    
    try {
        await showProgress('Creating production package.json');
        
        const packageJsonPath = path.join(rootDir, 'package.json');
        const packageJson = JSON.parse(fs.readFileSync(packageJsonPath, 'utf8'));
        
        // Remove devDependencies for production
        const productionPackageJson = {
            ...packageJson,
            devDependencies: undefined,
            scripts: {
                start: packageJson.scripts.start,
                migrate: packageJson.scripts.migrate,
                "migrate:create": packageJson.scripts["migrate:create"],
                "migrate:status": packageJson.scripts["migrate:status"],
                "migrate:up": packageJson.scripts["migrate:up"],
                "migrate:down": packageJson.scripts["migrate:down"],
                "migrate:reset": packageJson.scripts["migrate:reset"],
                "migrate:fresh": packageJson.scripts["migrate:fresh"]
            }
        };
        
        const distPackageJsonPath = path.join(rootDir, 'dist', 'package.json');
        fs.writeFileSync(distPackageJsonPath, JSON.stringify(productionPackageJson, null, 2));
        
        log.success('Production package.json created successfully!');
    } catch (error) {
        log.error('Failed to create production package.json:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create .env.example for production
async function createEnvExample() {
    log.header('CREATING PRODUCTION ENV EXAMPLE');
    
    try {
        await showProgress('Creating .env.example for production');
        
        const envExampleContent = `# Production Environment Configuration
# Copy this file to .env and configure your settings

# Server Configuration
SERVER_PORT=3000
ENVIRONMENT=production

# Security
SESSION_SECRET=your-super-secret-session-key-here
DISABLE_SECURE_COOKIES=false

# Database Configuration
DB_HOST=localhost
DB_USER=your-database-user
DB_PASSWORD=your-database-password
DB_DATABASE=your-database-name
DB_CONNECTION_LIMIT=500

# CORS Configuration
ALLOWED_ORIGINS=https://yourdomain.com,http://localhost:3000

# Rate Limiting
RATE_LIMIT_WINDOW_MS=900000
RATE_LIMIT_MAX_REQUESTS=1000
AUTH_RATE_LIMIT_WINDOW_MS=900000
AUTH_RATE_LIMIT_MAX=5

# Body Size Limits
BODY_SIZE_LIMIT=100mb
JSON_SIZE_LIMIT=10mb

# Email Configuration (Optional)
SMTP_HOST=
SMTP_PORT=587
SMTP_USER=
SMTP_PASSWORD=
SMTP_FROM_EMAIL=
SMTP_FROM_NAME=

# OAuth Configuration (Optional)
FACEBOOK_APP_ID=
FACEBOOK_APP_SECRET=
GOOGLE_CLIENT_ID=
GOOGLE_CLIENT_SECRET=

# Cache Configuration
CACHE_TTL=3600
CACHE_MAX_SIZE=1000

# Logging
LOG_LEVEL=info
LOG_FILE=logs/app.log

# Maintenance Mode
MAINTENANCE_MODE=false
MAINTENANCE_MESSAGE=Site is under maintenance. Please check back later.

# Game Configuration
MAX_GAME_SIZE=50000000
ALLOWED_GAME_TYPES=rom,tic80,flash,html5
`;

        const envExamplePath = path.join(rootDir, 'dist', '.env.example');
        fs.writeFileSync(envExamplePath, envExampleContent);
        
        log.success('.env.example created successfully!');
    } catch (error) {
        log.error('Failed to create .env.example:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create README for production
async function createProductionReadme() {
    log.header('CREATING PRODUCTION README');
    
    try {
        await showProgress('Creating production README');
        
        const readmeContent = `# Arcade Platform - Production Build

This is a production-ready build of the Arcade Platform.

## Quick Start

1. **Install Dependencies**
   \`\`\`bash
   npm install --production
   \`\`\`
   
   **Note:** This build does NOT include \`node_modules\` to keep the package size small and ensure clean dependencies. Dependencies will be installed fresh on the production server.

2. **Configure Environment**
   \`\`\`bash
   cp .env.example .env
   # Edit .env with your configuration
   \`\`\`

3. **Setup Database**
   \`\`\`bash
   # Create your database and run migrations
   npm run migrate:up
   \`\`\`

4. **Start the Application**
   \`\`\`bash
   npm start
   \`\`\`

## Configuration

### Environment Variables

Copy \`.env.example\` to \`.env\` and configure:

- **Database**: Set your MySQL connection details
- **Security**: Configure session secrets and CORS origins
- **Server**: Set port and environment settings
- **Email**: Configure SMTP for email functionality (optional)
- **OAuth**: Configure social login (optional)

### Database Setup

1. Create a MySQL database
2. Update \`.env\` with your database credentials
3. Run migrations: \`npm run migrate:up\`

### Production Deployment

For production deployment, consider:

- Using a process manager like PM2
- Setting up a reverse proxy (nginx)
- Configuring SSL certificates
- Setting up proper logging and monitoring

## File Structure

\`\`\`
dist/
├── app.js                 # Main application file
├── package.json           # Production dependencies
├── .env.example          # Environment configuration template
├── src/                   # Application source code
├── views/                 # EJS templates
├── public/                # Static assets
├── scripts/               # Utility scripts
├── migrations/            # Database migrations
└── uploads/               # User uploads directory
\`\`\`

## Support

For support and documentation, visit the project repository.

## License

This software is licensed under the terms specified in the original package.
`;

        const readmePath = path.join(rootDir, 'dist', 'README.md');
        fs.writeFileSync(readmePath, readmeContent);
        
        log.success('Production README created successfully!');
    } catch (error) {
        log.error('Failed to create production README:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create deployment script
async function createDeploymentScript() {
    log.header('CREATING DEPLOYMENT SCRIPT');
    
    try {
        await showProgress('Creating deployment script');
        
        const deployScriptContent = `#!/bin/bash

# Arcade Platform Deployment Script
# This script helps deploy the production build

set -e

echo "🚀 Starting Arcade Platform Deployment..."

# Check if Node.js is installed
if ! command -v node &> /dev/null; then
    echo "❌ Node.js is not installed. Please install Node.js first."
    exit 1
fi

# Check if npm is installed
if ! command -v npm &> /dev/null; then
    echo "❌ npm is not installed. Please install npm first."
    exit 1
fi

# Check if .env file exists
if [ ! -f ".env" ]; then
    echo "⚠️  .env file not found. Creating from template..."
    if [ -f ".env.example" ]; then
        cp .env.example .env
        echo "✅ .env file created from template. Please configure it before continuing."
        echo "📝 Edit .env file with your database and server configuration."
        exit 1
    else
        echo "❌ .env.example not found. Cannot create .env file."
        exit 1
    fi
fi

# Install production dependencies
echo "📦 Installing production dependencies..."
echo "   Note: node_modules are not included in this build for security and size reasons"
echo "   Dependencies will be installed fresh from package.json"
npm install --production

# Run database migrations
echo "🗄️  Running database migrations..."
npm run migrate:up

echo "✅ Deployment completed successfully!"
echo ""
echo "🎮 To start the application:"
echo "   npm start"
echo ""
echo "🌐 The application will be available at the port specified in your .env file."
echo "📊 Access the admin dashboard at: http://your-domain/dashboard"
`;

        const deployScriptPath = path.join(rootDir, 'dist', 'deploy.sh');
        fs.writeFileSync(deployScriptPath, deployScriptContent);
        
        // Make the script executable
        fs.chmodSync(deployScriptPath, '755');
        
        log.success('Deployment script created successfully!');
    } catch (error) {
        log.error('Failed to create deployment script:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Create archive
async function createArchive() {
    log.header('CREATING PRODUCTION ARCHIVE');
    
    try {
        await showProgress('Creating production archive');
        
        const archivePath = path.join(rootDir, 'arcade-platform-production.zip');
        const output = fs.createWriteStream(archivePath);
        const archive = archiver('zip', { zlib: { level: 9 } });
        
        return new Promise((resolve, reject) => {
            output.on('close', () => {
                log.success(`Archive created successfully! (${archive.pointer()} bytes)`);
                log.info(`Archive location: ${archivePath}`);
                resolve();
            });
            
            archive.on('error', (err) => {
                log.error('Archive creation failed:');
                console.log(`${colors.red}${err.message}${colors.reset}`);
                reject(err);
            });
            
            archive.pipe(output);
            archive.directory(path.join(rootDir, 'dist'), 'arcade-platform');
            archive.finalize();
        });
    } catch (error) {
        log.error('Failed to create archive:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        throw error;
    }
}

// Files and directories to exclude from the build
const excludePatterns = [
    'node_modules',
    '.git',
    '.env',
    'dist',
    '*.log',
    '*.tmp',
    '.DS_Store',
    'Thumbs.db',
    '*.zip',
    '*.tar.gz',
    'coverage',
    'tests',
    'documentation',
    '.vscode',
    '.idea',
    '*.swp',
    '*.swo',
    '*.orig',
    'package-lock.json'
];

// Check if a file/directory should be excluded
function shouldExclude(name) {
    return excludePatterns.some(pattern => {
        if (pattern.includes('*')) {
            // Handle wildcard patterns
            const regex = new RegExp(pattern.replace(/\*/g, '.*'));
            return regex.test(name);
        }
        return name === pattern;
    });
}

// Utility function to copy directories recursively with exclusions
async function copyDirectory(src, dest) {
    if (!fs.existsSync(dest)) {
        fs.mkdirSync(dest, { recursive: true });
    }
    
    const entries = fs.readdirSync(src, { withFileTypes: true });
    
    for (const entry of entries) {
        // Skip excluded files and directories
        if (shouldExclude(entry.name)) {
            continue;
        }
        
        const srcPath = path.join(src, entry.name);
        const destPath = path.join(dest, entry.name);
        
        if (entry.isDirectory()) {
            await copyDirectory(srcPath, destPath);
        } else {
            fs.copyFileSync(srcPath, destPath);
        }
    }
}

// Final setup steps
function finalizeBuild() {
    log.header('BUILD COMPLETE');
    
    log.highlight('🎉 PRODUCTION BUILD COMPLETED SUCCESSFULLY! 🎉');
    
    log.subheader('Build Summary');
    console.log(`  ${colors.bright}📁 Dist Directory:${colors.reset} ./dist/`);
    console.log(`  ${colors.bright}📦 Archive Created:${colors.reset} arcade-platform-production.zip`);
    console.log(`  ${colors.bright}🔧 Production Ready:${colors.reset} Yes`);
    console.log(`  ${colors.bright}📝 Documentation:${colors.reset} README.md included`);
    console.log(`  ${colors.bright}🚀 Deployment Script:${colors.reset} deploy.sh included`);
    console.log('');
    
    log.subheader('Next Steps');
    console.log(`  ${colors.bright}${colors.green}1.${colors.reset} Extract the archive to your production server`);
    console.log(`  ${colors.bright}${colors.green}2.${colors.reset} Configure your .env file with production settings`);
    console.log(`  ${colors.bright}${colors.green}3.${colors.reset} Run: ${colors.cyan}chmod +x deploy.sh && ./deploy.sh${colors.reset}`);
    console.log(`  ${colors.bright}${colors.green}4.${colors.reset} Start the application: ${colors.cyan}npm start${colors.reset}`);
    console.log('');
    
    log.subheader('Production Features');
    console.log(`  ${colors.bright}✓${colors.reset} Optimized assets and minified code`);
    console.log(`  ${colors.bright}✓${colors.reset} Production-only dependencies`);
    console.log(`  ${colors.bright}✓${colors.reset} Environment configuration template`);
    console.log(`  ${colors.bright}✓${colors.reset} Database migration system`);
    console.log(`  ${colors.bright}✓${colors.reset} Automated deployment script`);
    console.log(`  ${colors.bright}✓${colors.reset} Complete documentation`);
    console.log('');
}

// Main build function
async function build() {
    const startTime = Date.now();
    
    try {
        showWelcomeBanner();
        
        // Build steps
        const steps = [
            { name: 'Clean Dist Directory', fn: cleanDistDirectory },
            { name: 'Create Directory Structure', fn: createDistStructure },
            { name: 'Copy Essential Files', fn: copyEssentialFiles },
            { name: 'Copy Source Code', fn: copySourceCode },
            { name: 'Copy Public Assets', fn: copyPublicAssets },
            { name: 'Copy Uploads Structure', fn: copyUploadsStructure },
            { name: 'Create Production Package.json', fn: createProductionPackageJson },
            { name: 'Create Environment Example', fn: createEnvExample },
            { name: 'Create Production README', fn: createProductionReadme },
            { name: 'Create Deployment Script', fn: createDeploymentScript },
            { name: 'Create Production Archive', fn: createArchive }
        ];
        
        console.log('');
        log.info(`Starting build process with ${steps.length} steps...`);
        console.log('');
        
        for (let i = 0; i < steps.length; i++) {
            const step = steps[i];
            const stepStartTime = Date.now();
            
            log.step(`[${i + 1}/${steps.length}] ${step.name}`);
            
            try {
                await step.fn();
                const stepDuration = Date.now() - stepStartTime;
                log.success(`Step completed: ${step.name} (${stepDuration}ms)`);
                console.log('');
            } catch (stepError) {
                const stepDuration = Date.now() - stepStartTime;
                log.error(`Step failed: ${step.name} (${stepDuration}ms)`);
                log.error(`Error: ${stepError.message}`);
                throw stepError;
            }
        }
        
        // Calculate total build time
        const totalDuration = Date.now() - startTime;
        const minutes = Math.floor(totalDuration / 60000);
        const seconds = Math.floor((totalDuration % 60000) / 1000);
        
        finalizeBuild();
        
        log.success(`Build completed successfully in ${minutes}m ${seconds}s!`);
        console.log('');
        
    } catch (error) {
        const totalDuration = Date.now() - startTime;
        const minutes = Math.floor(totalDuration / 60000);
        const seconds = Math.floor((totalDuration % 60000) / 1000);
        
        log.error(`Build failed after ${minutes}m ${seconds}s:`);
        console.log(`${colors.red}${error.message}${colors.reset}`);
        console.log('');
        
        console.log(`${colors.yellow}Build Troubleshooting:${colors.reset}`);
        console.log(`  - Check the error message above for specific details`);
        console.log(`  - Ensure all source files are present and accessible`);
        console.log(`  - Verify file permissions for read/write operations`);
        console.log(`  - Check available disk space for the build process`);
        console.log('');
        
        process.exit(1);
    }
}

// Show help information
function showHelp() {
    console.log('');
    console.log(`${colors.bright}${colors.cyan}Build Script Help${colors.reset}`);
    console.log('');
    console.log(`${colors.bright}Usage:${colors.reset}`);
    console.log(`  node scripts/build.js [options]`);
    console.log('');
    console.log(`${colors.bright}Options:${colors.reset}`);
    console.log(`  --help, -h    Show this help message`);
    console.log('');
    console.log(`${colors.bright}Description:${colors.reset}`);
    console.log(`  Creates a production-ready distribution package of the Arcade Platform.`);
    console.log(`  This includes optimized assets, production configuration, and deployment scripts.`);
    console.log('');
    console.log(`${colors.bright}Output:${colors.reset}`);
    console.log(`  • dist/ directory with production files`);
    console.log(`  • arcade-platform-production.zip archive`);
    console.log(`  • Production-ready package.json`);
    console.log(`  • Environment configuration template`);
    console.log(`  • Deployment documentation and scripts`);
    console.log('');
}

// Parse command line arguments
const args = process.argv.slice(2);
if (args.includes('--help') || args.includes('-h')) {
    showHelp();
    process.exit(0);
}

// Run build
build()
    .catch(error => {
        console.error(error);
        process.exit(1);
    });
