#!/usr/bin/env node

import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import mysql from 'mysql2/promise';
import crypto from 'crypto';
import bcrypt from 'bcrypt';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const rootDir = path.resolve(__dirname, '..');

// Colors for beautiful CLI output
const colors = {
    reset: '\x1b[0m',
    bright: '\x1b[1m',
    dim: '\x1b[2m',
    red: '\x1b[31m',
    green: '\x1b[32m',
    yellow: '\x1b[33m',
    blue: '\x1b[34m',
    magenta: '\x1b[35m',
    cyan: '\x1b[36m'
};

// Styled console functions
const log = {
    success: (msg) => console.log(`${colors.green}✓${colors.reset} ${msg}`),
    error: (msg) => console.log(`${colors.red}✗${colors.reset} ${msg}`),
    warning: (msg) => console.log(`${colors.yellow}⚠${colors.reset} ${msg}`),
    info: (msg) => console.log(`${colors.blue}ℹ${colors.reset} ${msg}`),
    step: (msg) => console.log(`${colors.cyan}→${colors.reset} ${msg}`),
    header: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.cyan}═══════════════════════════════════════════════════════════════════════════════════${colors.reset}`);
        console.log(`${colors.bright}${colors.cyan}  ${msg}  ${colors.reset}`);
        console.log(`${colors.bright}${colors.cyan}═══════════════════════════════════════════════════════════════════════════════════${colors.reset}`);
        console.log('');
    },
    subheader: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.yellow}${msg}${colors.reset}`);
        console.log(`${colors.bright}${colors.blue}${'─'.repeat(msg.length)}${colors.reset}`);
        console.log('');
    },
    highlight: (msg) => {
        console.log('');
        console.log(`${colors.bright}${colors.magenta}${'═'.repeat(80)}${colors.reset}`);
        console.log(`${colors.bright}${colors.magenta}  ${msg}  ${colors.reset}`);
        console.log(`${colors.bright}${colors.magenta}${'═'.repeat(80)}${colors.reset}`);
        console.log('');
    }
};

// Progress indicator without actual delay
function showProgress(message) {
    process.stdout.write(`${colors.cyan}⠋${colors.reset} ${message}...`);
    // Simulate brief processing
    setTimeout(() => {
        process.stdout.write(`\r${colors.green}✓${colors.reset} ${message}\n`);
    }, 100);
    return new Promise(resolve => setTimeout(resolve, 150));
}

// Parse command line arguments
function parseArguments() {
    const args = process.argv.slice(2);
    
    if (args.includes('--help') || args.includes('-h')) {
        showHelp();
        process.exit(0);
    }
    
    const config = {
        site: {
            name: getArgValue(args, '--site-name') || 'Arcade Platform',
            description: getArgValue(args, '--site-description') || 'Gaming Platform',
            port: getArgValue(args, '--port') || '80',
            url: getArgValue(args, '--site-url') || 'http://localhost',
            purchaseCode: getArgValue(args, '--purchase-code') || ''
        },
        admin: {
            username: getArgValue(args, '--admin-username') || 'admin',
            firstName: getArgValue(args, '--admin-first-name') || 'Admin',
            lastName: getArgValue(args, '--admin-last-name') || 'User',
            email: getArgValue(args, '--admin-email') || 'admin@localhost',
            password: getArgValue(args, '--admin-password') || 'admin123'
        },
        db: {
            host: getArgValue(args, '--db-host') || 'localhost',
            name: getArgValue(args, '--db-name') || 'arcade',
            username: getArgValue(args, '--db-username') || 'server',
            password: getArgValue(args, '--db-password') || 'password'
        }
    };
    
    return config;
}

function getArgValue(args, flag) {
    const index = args.indexOf(flag);
    return index !== -1 && index + 1 < args.length ? args[index + 1] : null;
}

// Show help information
function showHelp() {
    console.log('');
    console.log(`${colors.bright}${colors.cyan}Automated Deployment Script Help${colors.reset}`);
    console.log('');
    console.log(`${colors.bright}Usage:${colors.reset}`);
    console.log(`  node scripts/deploy_vps.js [options]`);
    console.log('');
    console.log(`${colors.bright}Options:${colors.reset}`);
    console.log(`  --site-name <name>         Site name (default: "Arcade Platform")`);
    console.log(`  --site-description <desc>  Site description (default: "Gaming Platform")`);
    console.log(`  --purchase-code <code>     Purchase code for license verification (required)`);
    console.log(`  --port <port>              Server port (default: "80")`);
    console.log(`  --site-url <url>           Site URL (default: "http://localhost")`);
    console.log(`  --admin-username <user>    Admin username (default: "admin")`);
    console.log(`  --admin-first-name <name>  Admin first name (default: "Admin")`);
    console.log(`  --admin-last-name <name>   Admin last name (default: "User")`);
    console.log(`  --admin-email <email>      Admin email (default: "admin@localhost")`);
    console.log(`  --admin-password <pass>    Admin password (default: "admin123")`);
    console.log(`  --db-host <host>           Database host (default: "localhost")`);
    console.log(`  --db-name <name>           Database name (default: "arcade")`);
    console.log(`  --db-username <user>       Database username (default: "server")`);
    console.log(`  --db-password <pass>       Database password (required)`);
    console.log(`  --help, -h                 Show this help message`);
    console.log('');
    console.log(`${colors.bright}Description:${colors.reset}`);
    console.log(`  Fully automated deployment script for the Titan Systems Arcade Platform.`);
    console.log(`  This script performs the same actions as deploy.js but without user interaction.`);
    console.log('');
    console.log(`${colors.bright}Example:${colors.reset}`);
    console.log(`  node scripts/deploy_vps.js \\`);
    console.log(`    --site-name "My Arcade" \\`);
    console.log(`    --site-url "https://arcade.example.com" \\`);
    console.log(`    --admin-email "admin@example.com" \\`);
    console.log(`    --admin-password "securepassword123" \\`);
    console.log(`    --purchase-code "ABCD123456-XYZ" \\`);
    console.log(`    --db-password "mysqlpassword"`);
    console.log('');
}

// Welcome banner
function showWelcomeBanner() {
    console.log('');
    console.log(`${colors.bright}${colors.cyan}                  🚀 AUTOMATED ARCADE DEPLOYMENT 🚀${colors.reset}`);
    console.log(`${colors.bright}${colors.yellow}              ✨ Fully Automated Setup Process ✨${colors.reset}`);
    console.log('');
    console.log(`${colors.dim}   Automated deployment of the Titan Systems Arcade Platform${colors.reset}`);
    console.log(`${colors.dim}   This script will configure your platform without user interaction${colors.reset}`);
    console.log('');
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Secure environment configuration${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Database schema installation${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Admin account creation${colors.reset}`);
    console.log(`     ${colors.green}✓${colors.reset} ${colors.bright}Production-ready setup${colors.reset}`);
    console.log('');
}

// Validate configuration
function validateConfig(config) {
    const errors = [];
    
    // Site validation
    if (!config.site.name || config.site.name.length < 3) {
        errors.push('Site name must be at least 3 characters long');
    }
    
    if (!config.site.description || config.site.description.length < 3) {
        errors.push('Site description must be at least 3 characters long');
    }
    
    if (!config.site.purchaseCode) {
        errors.push('Purchase code is required');
    } else if (!/^[a-zA-Z0-9-]+$/.test(config.site.purchaseCode)) {
        errors.push('Purchase code must contain only letters, numbers, and hyphens');
    }
    
    const port = parseInt(config.site.port);
    if (isNaN(port) || port < 1 || port > 65535) {
        errors.push('Port must be a number between 1 and 65535');
    }
    
    // Admin validation
    if (!config.admin.username.match(/^[a-zA-Z0-9_-]{3,20}$/)) {
        errors.push('Admin username must be 3-20 characters long and contain only letters, numbers, hyphens, and underscores');
    }
    
    if (!config.admin.email.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
        errors.push('Admin email must be a valid email address');
    }
    
    if (!config.admin.password || config.admin.password.length < 6) {
        errors.push('Admin password must be at least 6 characters long');
    }
    
    // Database validation
    if (!config.db.name.match(/^[a-zA-Z0-9_-]{1,64}$/)) {
        errors.push('Database name must be 1-64 characters long and contain only letters, numbers, hyphens, and underscores');
    }
    
    if (!config.db.username.match(/^[a-zA-Z0-9_-]{3,20}$/)) {
        errors.push('Database username must be 3-20 characters long and contain only letters, numbers, hyphens, and underscores');
    }
    
    if (!config.db.password) {
        errors.push('Database password is required');
    }
    
    return errors;
}

// Test database connection with timeout
async function testDatabaseConnection(config) {
    log.step('Testing database connection');
    await showProgress('Connecting to database with 30-second timeout');

    try {
        const connection = await mysql.createConnection({
            host: config.db.host,
            port: 3306,
            user: config.db.username,
            password: config.db.password,
            database: config.db.name,
            connectTimeout: 30000,    // 30 seconds
            acquireTimeout: 15000,    // 15 seconds
            timeout: 60000           // 60 seconds for queries
        });

        // Test with a simple query that has timeout
        await Promise.race([
            connection.execute('SELECT 1 as test'),
            new Promise((_, reject) =>
                setTimeout(() => reject(new Error('Query timeout after 10 seconds')), 10000)
            )
        ]);

        await connection.end();

        log.success('Database connection successful!');
        return true;
    } catch (error) {
        log.error('Database connection failed:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        console.log(`${colors.yellow}Troubleshooting tips:${colors.reset}`);
        console.log(`  - Check if MySQL service is running: systemctl status mysql`);
        console.log(`  - Verify database credentials and host accessibility`);
        console.log(`  - Ensure no firewall blocking database port 3306`);
        return false;
    }
}

// Generate .env file from template
async function generateEnvFile(config) {
    log.header('GENERATING CONFIGURATION');
    await showProgress('Reading .env.example template');
    
    try {
        const envExamplePath = path.join(rootDir, '.env.example');
        let envContent = fs.readFileSync(envExamplePath, 'utf8');
        
        // Generate unique session secret
        const sessionSecret = crypto.randomBytes(64).toString('hex');
        
        // Replace template values
        envContent = envContent
            .replace(/SERVER_PORT=.*/, `SERVER_PORT=${config.site.port}`)
            .replace(/SESSION_SECRET=.*/, `SESSION_SECRET=${sessionSecret}`)
            .replace(/ENVIRONMENT=.*/, `ENVIRONMENT=production`)
            .replace(/ALLOWED_ORIGINS=.*/, `ALLOWED_ORIGINS=${config.site.url}`)
            .replace(/DB_HOST=.*/, `DB_HOST=${config.db.host}`)
            .replace(/DB_USER=.*/, `DB_USER=${config.db.username}`)
            .replace(/DB_PASSWORD=.*/, `DB_PASSWORD=${config.db.password}`)
            .replace(/DB_DATABASE=.*/, `DB_DATABASE=${config.db.name}`);
        
        // Note: Purchase code is stored in database only, not in .env file
        
        // Add missing fields if not present
        if (!envContent.includes('DB_CONNECTION_LIMIT')) {
            envContent += '\nDB_CONNECTION_LIMIT=500';
        }

        // Ensure DISABLE_SECURE_COOKIES is preserved for session security fallback
        if (!envContent.includes('DISABLE_SECURE_COOKIES')) {
            envContent += '\nDISABLE_SECURE_COOKIES=false';
        }
        
        await showProgress('Writing .env file');
        const envPath = path.join(rootDir, '.env');
        fs.writeFileSync(envPath, envContent);
        
        log.success('.env file created successfully!');
        return true;
    } catch (error) {
        log.error('Failed to create .env file:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        return false;
    }
}

// Install database base schema (migration table only)
async function installDatabase(config) {
    log.header('DATABASE INITIALIZATION');
    await showProgress('Reading db_schema.sql');
    
    try {
        const schemaPath = path.join(rootDir, 'db_schema.sql');
        const schemaContent = fs.readFileSync(schemaPath, 'utf8');
        
        await showProgress('Connecting to database with timeout settings');
        const connection = await mysql.createConnection({
            host: config.db.host,
            port: 3306,
            user: config.db.username,
            password: config.db.password,
            database: config.db.name,
            multipleStatements: true,
            connectTimeout: 30000,    // 30 seconds
            acquireTimeout: 15000,    // 15 seconds
            timeout: 120000          // 120 seconds for schema operations
        });
        
        await showProgress('Installing migration system');
        
        // Split SQL file into individual statements and execute them separately
        const statements = schemaContent
            .split(';')
            .map(stmt => stmt.trim())
            .filter(stmt => stmt.length > 0 && !stmt.startsWith('--'));
        
        for (const statement of statements) {
            if (statement) {
                await connection.execute(statement);
            }
        }
        
        await connection.end();
        
        log.success('Migration system initialized successfully!');
        console.log('');
        console.log(`  ${colors.bright}📝 Migration-based deployment:${colors.reset}`);
        console.log(`     • Database tables will be created via migrations`);
        console.log(`     • Admin account will be created via migrations`);
        console.log(`     • This ensures consistent deployments and easy updates`);
        console.log('');
        
        return true;
    } catch (error) {
        log.error('Database initialization failed:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        return false;
    }
}

// Check and run pending migrations with timeout protection
async function runMigrations() {
    log.header('MIGRATION SYSTEM');

    try {
        // Load environment variables from the newly created .env file
        await showProgress('Loading environment configuration');
        const envPath = path.join(rootDir, '.env');
        const envContent = fs.readFileSync(envPath, 'utf8');

        // Parse and set environment variables
        envContent.split('\n').forEach(line => {
            const trimmed = line.trim();
            if (trimmed && !trimmed.startsWith('#') && trimmed.includes('=')) {
                const [key, ...valueParts] = trimmed.split('=');
                const value = valueParts.join('=');
                // Only set if key is not empty
                if (key) {
                    process.env[key] = value;
                }
            }
        });

        // Ensure critical environment variables are set
        if (!process.env.DB_CONNECTION_LIMIT) {
            process.env.DB_CONNECTION_LIMIT = '500';
        }

        await showProgress('Loading migration utilities with timeout protection');

        // Dynamic import of migration utilities with timeout
        const migrationModule = await Promise.race([
            import('../src/utils/migration.js'),
            new Promise((_, reject) =>
                setTimeout(() => reject(new Error('Migration module import timeout after 30 seconds')), 30000)
            )
        ]);

        const { getMigrationStatusInfo, runPendingMigrations } = migrationModule;

        log.step('Checking migration status...');
        const status = await Promise.race([
            getMigrationStatusInfo(),
            new Promise((_, reject) =>
                setTimeout(() => reject(new Error('Migration status check timeout after 30 seconds')), 30000)
            )
        ]);

        console.log('');
        console.log(`  ${colors.bright}📁 Migration files:${colors.reset} ${status.totalMigrationFiles}`);
        console.log(`  ${colors.bright}✅ Applied:${colors.reset} ${status.appliedMigrations}`);
        console.log(`  ${colors.bright}⏳ Pending:${colors.reset} ${status.pendingMigrations}`);

        if (status.pendingMigrations > 0) {
            console.log('');
            log.warning(`Found ${status.pendingMigrations} pending migrations`);
            console.log(`  ${colors.dim}These migrations will create all database tables and setup data${colors.reset}`);

            // For fresh deployments, automatically run all migrations
            console.log('');
            log.step('Running all pending migrations automatically...');
            log.info(`This may take a few minutes depending on database size...`);

            // Show progress indicator for migrations
            process.stdout.write(`${colors.cyan}⠋${colors.reset} Executing migrations...`);
            const progressInterval = setInterval(() => {
                process.stdout.write('.');
            }, 2000);

            try {
                // Run migrations with extended timeout (8 minutes)
                const result = await Promise.race([
                    runPendingMigrations(),
                    new Promise((_, reject) =>
                        setTimeout(() => reject(new Error('Migration execution timeout after 8 minutes')), 480000)
                    )
                ]);

                clearInterval(progressInterval);
                process.stdout.write(`\r${colors.green}✓${colors.reset} Migrations completed\n`);

                if (result.success) {
                    console.log('');
                    result.results.forEach(migration => {
                        console.log(`  ${colors.green}✓${colors.reset} ${migration.migrationName} ${colors.dim}(${migration.executionTime}ms)${colors.reset}`);
                    });
                    console.log('');
                    log.success(`Successfully applied ${result.results.length} migrations!`);
                } else {
                    throw new Error(result.message);
                }
            } catch (migrationError) {
                clearInterval(progressInterval);
                process.stdout.write(`\r${colors.red}✗${colors.reset} Migration failed\n`);
                throw migrationError;
            }

        } else {
            console.log('');
            log.success('All migrations are up to date!');
        }

        return true;
    } catch (error) {
        log.error('Migration system failed:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        console.log('');
        console.log(`${colors.yellow}Common solutions:${colors.reset}`);
        console.log(`  - Check database connectivity and credentials`);
        console.log(`  - Ensure migration files are not corrupted`);
        console.log(`  - Verify sufficient database permissions`);
        console.log(`  - Check disk space for database operations`);
        return false;
    }
}

// Update admin account after migrations are complete
async function updateAdminAccount(config) {
    log.header('ADMIN ACCOUNT SETUP');
    
    try {
        await showProgress('Connecting to database for admin setup');
        const connection = await mysql.createConnection({
            host: config.db.host,
            port: 3306,
            user: config.db.username,
            password: config.db.password,
            database: config.db.name,
            connectTimeout: 30000,    // 30 seconds
            acquireTimeout: 15000,    // 15 seconds
            timeout: 60000           // 60 seconds for queries
        });
        
        // Create admin account
        await showProgress('Updating admin account');
        const hashedPassword = await bcrypt.hash(config.admin.password, 12);
        
        // Remove the default admin account first
        await connection.execute('DELETE FROM users WHERE id = 1');
        
        // Insert new admin account
        await connection.execute(`
            INSERT INTO users (username, email, password, first_name, last_name, user_type, level, exp_points, total_exp_earned, is_active, is_verified, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, 'admin', 1, 0, 0, 1, 1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
        `, [config.admin.username, config.admin.email, hashedPassword, config.admin.firstName, config.admin.lastName]);
        
        // Update site settings
        await showProgress('Updating site settings');
        await connection.execute('UPDATE settings SET value = ? WHERE name = "site_name"', [config.site.name]);
        await connection.execute('UPDATE settings SET value = ? WHERE name = "site_description"', [config.site.description]);
        await connection.execute('UPDATE settings SET value = ? WHERE name = "purchase_code"', [config.site.purchaseCode]);
        
        await connection.end();
        
        log.success('Admin account configured successfully!');
        return true;
    } catch (error) {
        log.error('Admin account setup failed:');
        console.log(`${colors.red}${error.message}${colors.reset}`);
        return false;
    }
}

// Final setup steps
function finalizeSetup(config) {
    log.header('DEPLOYMENT COMPLETE');
    
    log.highlight('✅ DATABASE SETUP AND CONFIGURATION COMPLETE! ✅');
    
    log.subheader('Configuration Summary');
    console.log(`  ${colors.bright}🏷️  Site Name:${colors.reset} ${config.site.name}`);
    console.log(`  ${colors.bright}📝 Description:${colors.reset} ${config.site.description}`);
    console.log(`  ${colors.bright}🌐 Server Port:${colors.reset} ${config.site.port}`);
    console.log(`  ${colors.bright}🌍 Site URL(s):${colors.reset} ${config.site.url}`);
    console.log(`  ${colors.bright}👤 Admin Username:${colors.reset} ${config.admin.username}`);
    console.log(`  ${colors.bright}📧 Admin Email:${colors.reset} ${config.admin.email}`);
    console.log(`  ${colors.bright}🗄️  Database:${colors.reset} ${config.db.name} at ${config.db.host}`);
    console.log(`  ${colors.bright}🔧 Environment:${colors.reset} production`);
    console.log('');
    
    log.subheader('Application Ready');
    console.log(`  ${colors.bright}${colors.green}✓${colors.reset} .env file created`);
    console.log(`  ${colors.bright}${colors.green}✓${colors.reset} Database initialized`);
    console.log(`  ${colors.bright}${colors.green}✓${colors.reset} Migrations applied`);
    console.log(`  ${colors.bright}${colors.green}✓${colors.reset} Admin account created`);
    console.log(`  ${colors.bright}${colors.green}✓${colors.reset} Site settings configured`);
    console.log('');
    
    log.subheader('Next Steps');
    console.log(`  ${colors.bright}${colors.yellow}→${colors.reset} VPS setup script will now configure PM2 process manager`);
    console.log(`  ${colors.bright}${colors.yellow}→${colors.reset} Application service will be started automatically`);
    console.log(`  ${colors.bright}${colors.yellow}→${colors.reset} Final credentials and access info will be displayed`);
    console.log('');
}

// Main deployment function with comprehensive error handling
async function deployAuto() {
    const startTime = Date.now();

    try {
        const config = parseArguments();

        showWelcomeBanner();

        log.header('CONFIGURATION VALIDATION');
        console.log(`  ${colors.bright}Site Name:${colors.reset} ${config.site.name}`);
        console.log(`  ${colors.bright}Site URL:${colors.reset} ${config.site.url}`);
        console.log(`  ${colors.bright}Purchase Code:${colors.reset} ${config.site.purchaseCode ? config.site.purchaseCode.substring(0, 10) + '...' : 'Not provided'}`);
        console.log(`  ${colors.bright}Server Port:${colors.reset} ${config.site.port}`);
        console.log(`  ${colors.bright}Admin Username:${colors.reset} ${config.admin.username}`);
        console.log(`  ${colors.bright}Admin Email:${colors.reset} ${config.admin.email}`);
        console.log(`  ${colors.bright}Database:${colors.reset} ${config.db.name}@${config.db.host}`);
        console.log('');

        // Validate configuration
        const errors = validateConfig(config);
        if (errors.length > 0) {
            log.error('Configuration validation failed:');
            errors.forEach(error => console.log(`  ${colors.red}✗${colors.reset} ${error}`));
            process.exit(1);
        }

        log.success('Configuration validation passed!');

        // Deployment steps with progress tracking
        const steps = [
            { name: 'Database Connection Test', fn: () => testDatabaseConnection(config) },
            { name: 'Environment Configuration', fn: () => generateEnvFile(config) },
            { name: 'Database Schema Installation', fn: () => installDatabase(config) },
            { name: 'Migration System Execution', fn: () => runMigrations() },
            { name: 'Admin Account Configuration', fn: () => updateAdminAccount(config) }
        ];

        console.log('');
        log.info(`Starting deployment with ${steps.length} main steps...`);
        console.log('');

        for (let i = 0; i < steps.length; i++) {
            const step = steps[i];
            const stepStartTime = Date.now();

            log.step(`[${i + 1}/${steps.length}] ${step.name}`);

            try {
                const success = await step.fn();
                const stepDuration = Date.now() - stepStartTime;

                if (!success) {
                    log.error(`Step failed: ${step.name} (${stepDuration}ms)`);
                    process.exit(1);
                }

                log.success(`Step completed: ${step.name} (${stepDuration}ms)`);
                console.log('');
            } catch (stepError) {
                const stepDuration = Date.now() - stepStartTime;
                log.error(`Step failed with exception: ${step.name} (${stepDuration}ms)`);
                log.error(`Error: ${stepError.message}`);
                throw stepError;
            }
        }

        // Calculate total deployment time
        const totalDuration = Date.now() - startTime;
        const minutes = Math.floor(totalDuration / 60000);
        const seconds = Math.floor((totalDuration % 60000) / 1000);

        finalizeSetup(config);

        log.success(`Database deployment completed successfully in ${minutes}m ${seconds}s!`);
        console.log('');
        log.info('Returning to VPS setup script for service configuration...');
        console.log('');

        // Debug: Show that we're about to exit
        log.step('Cleaning up and exiting deployment script...');

        // Force clean exit immediately
        process.exit(0);

    } catch (error) {
        const totalDuration = Date.now() - startTime;
        const minutes = Math.floor(totalDuration / 60000);
        const seconds = Math.floor((totalDuration % 60000) / 1000);

        log.error(`Automated deployment failed after ${minutes}m ${seconds}s:`);
        console.log(`${colors.red}${error.message}${colors.reset}`);

        // Enhanced error reporting
        if (error.code) {
            console.log(`${colors.yellow}Error Code: ${error.code}${colors.reset}`);
        }

        console.log('');
        console.log(`${colors.yellow}Deployment Troubleshooting:${colors.reset}`);
        console.log(`  - Check the error message above for specific details`);
        console.log(`  - Verify all prerequisites are installed and configured`);
        console.log(`  - Ensure database service is running and accessible`);
        console.log(`  - Check network connectivity and firewall settings`);
        console.log(`  - Review the error stack trace below for technical details`);
        console.log('');

        // Only show stack trace in verbose mode or for certain errors
        if (process.env.VERBOSE || error.name === 'TimeoutError') {
            console.log(`${colors.dim}Stack trace:${colors.reset}`);
            console.error(error.stack);
        }

        process.exit(1);
    }
}

// Run deployment
deployAuto()
    .catch(error => {
        console.error(error);
        process.exit(1);
    });