import fs from 'fs/promises';
import path from 'path';
import { consoleLog } from '../utils/logger.js';
import response from '../utils/response.js';
import i18n from '../utils/i18n.js';

const pluginsCache = new Map();
const pluginsDir = path.join(process.cwd(), 'plugins');

/**
 * Plugin error handler (like PHP pluginError method)
 * @param {object} res - Express response object
 * @param {string} message - Error message
 * @param {boolean} json - Whether to return JSON response
 */
const pluginError = (res, message, json = false) => {
  if (json) {
    return response(res, 500, message);
  } else {
    return res.redirect('/errors/500');
  }
};

/**
 * Load plugin dynamically (exactly like PHP approach)
 * @param {string} pluginName - Name of the plugin to load
 * @returns {object} Plugin object with info, actions, and path
 */
async function loadPlugin(pluginName) {
  try {
    const pluginPath = path.join(pluginsDir, pluginName);
    const infoPath = path.join(pluginPath, 'info.json');
    const mainPath = path.join(pluginPath, 'plugin.js');

    // Check if info.json exists
    const infoExists = await fs.access(infoPath).then(() => true).catch(() => false);
    if (!infoExists) {
      throw new Error(`Plugin info file not found: ${pluginName}/info.json`);
    }

    // Check if plugin.js exists
    const mainExists = await fs.access(mainPath).then(() => true).catch(() => false);
    if (!mainExists) {
      throw new Error(`Plugin file not found: ${pluginName}/plugin.js`);
    }

    // Read plugin metadata
    const infoContent = await fs.readFile(infoPath, 'utf8');
    const pluginData = JSON.parse(infoContent);

    // Check if plugin is enabled
    if (!pluginData.enabled) {
      throw new Error(`Plugin ${pluginName} is disabled`);
    }

    // Load models if specified (like PHP models loading)
    if (pluginData.models && pluginData.models.length > 0) {
      for (const model of pluginData.models) {
        const modelPath = path.join(pluginPath, 'models', `${model}.js`);
        try {
          await import(`${modelPath}?t=${Date.now()}`);
          consoleLog('plugin', `Loaded model: ${pluginName}/${model}`);
        } catch (error) {
          throw new Error(`Failed to load plugin model: ${model} - ${error.message}`);
        }
      }
    }

    // Import plugin actions with cache busting (like PHP require)
    const cacheBuster = Date.now();
    const pluginModule = await import(`${mainPath}?t=${cacheBuster}`);
    const pluginActions = pluginModule.default;

    if (!pluginActions || typeof pluginActions !== 'object') {
      throw new Error(`Plugin ${pluginName} must export a default object with action functions`);
    }

    consoleLog('plugin', `Plugin loaded successfully: ${pluginName}`);

    return {
      info: pluginData,
      actions: pluginActions,
      path: pluginPath
    };
  } catch (error) {
    consoleLog('error', `Failed to load plugin ${pluginName}`, { error: error.message });
    throw error;
  }
}

/**
 * Main plugin handler (exactly like PHP index method)
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 */
const pluginHandler = async (req, res) => {
  // Get parameters (like PHP sanitize->array($_REQUEST))
  const request = {
    ...req.query,
    ...req.body,
    ...req.params
  };

  const json = request.json ? true : false;

  // Check if plugin name is provided
  if (!request.name) {
    return pluginError(res, i18n.translateSync('errors.plugin_went_wrong', {}, req.language?.current || 'en'), json);
  }

  // Set default action if not provided
  if (!request.action || request.action.trim() === '') {
    request.action = 'default';
  }

  try {
    // Load plugin (with caching like the system)
    let plugin = pluginsCache.get(request.name);
    
    // Reload in development or if not cached
    if (!plugin || process.env.NODE_ENV === 'development') {
      plugin = await loadPlugin(request.name);
      pluginsCache.set(request.name, plugin);
    }

    // Check if action exists in plugin (like PHP isset check)
    if (!plugin.actions[request.action]) {
      return pluginError(res, `Action '${request.action}' not found in plugin '${request.name}'`, json);
    }

    // Call the plugin action (exactly like PHP approach)
    try {
      await plugin.actions[request.action](req, res, request);
    } catch (error) {
      consoleLog('error', `Plugin action error: ${request.name}.${request.action}`, { error: error.message });
      return pluginError(res, `Plugin Error: ${error.message}`, json);
    }

  } catch (error) {
    consoleLog('error', `Plugin system error: ${request.name}`, { error: error.message });
    return pluginError(res, error.message, json);
  }
};

/**
 * Plugin asset handler
 * @param {object} req - Express request object
 * @param {object} res - Express response object
 */
const pluginAssets = async (req, res) => {
  const pluginName = req.query.name;
  
  if (!pluginName) {
    return res.status(400).send('Plugin name required');
  }
  
  try {
    // Get plugin (load if not cached)
    let plugin = pluginsCache.get(pluginName);
    if (!plugin) {
      plugin = await loadPlugin(pluginName);
      pluginsCache.set(pluginName, plugin);
    }
    
    const assetsPath = path.join(plugin.path, 'assets');
    
    // Remove the /plugin/assets/ part and get the actual file path
    const filePath = req.path.replace('/assets/', '');
    const fullPath = path.join(assetsPath, filePath);
    
    // Check if file exists
    const fileExists = await fs.access(fullPath).then(() => true).catch(() => false);
    if (!fileExists) {
      return res.status(404).send('Asset not found');
    }
    
    // Serve the file
    return res.sendFile(fullPath);
  } catch (error) {
    consoleLog('error', `Plugin asset error: ${pluginName}`, { error: error.message });
    return res.status(404).send('Plugin assets not found');
  }
};

export {
  pluginHandler,
  pluginAssets
};