import cache from '@titansys/persistent-cache';
import { consoleLog } from './logger.js';

/**
 * All-in-one cache utility
 */
const cacheInstances = new Map();
const cacheUsageTracking = new Map(); // Track last access time for each cache
const healthMonitoring = new Map(); // Track health monitoring status

const CacheUtils = {
  /**
   * Get cache-type-specific configuration for optimal performance
   * @param {string} cacheName - Cache name
   * @param {number} duration - Base duration in milliseconds
   * @returns {Object} Cache configuration object
   */
  getCacheConfig(cacheName, duration) {
    // Tier 1: High Performance caches (memory + disk hybrid)
    const highPerformanceCaches = [
      'homepage-games', 'featured-games', 'i18n-languages', 'i18n-metadata',
      'sidebar-categories', 'popular-games', 'trending-games', 'recent-games',
      'top-rated-games', 'homepage-randomized-categories'
    ];

    // Tier 2: Standard performance caches (disk-based with standard monitoring)
    const standardCaches = [
      'ads', 'ad-placements', 'game-stats', 'popular-tags', 'top-searches'
    ];

    // Tier 3: Archive caches (minimal memory, longer health intervals)
    const archiveCaches = [
      'cronjob-logs', 'analytics', 'email-logs', 'error-logs'
    ];

    const baseConfig = {
      name: cacheName,
      duration: duration,
      base: './cache',
      persist: true,
      autoRepair: true,
      lockTimeout: 10000
    };

    if (highPerformanceCaches.includes(cacheName)) {
      return {
        ...baseConfig,
        memory: true, // Hybrid memory + disk for performance
        healthCheckInterval: 300000, // 5-minute health checks
        maxRetries: 3
      };
    } else if (standardCaches.includes(cacheName)) {
      return {
        ...baseConfig,
        memory: false, // Disk-based storage
        healthCheckInterval: 600000, // 10-minute health checks
        maxRetries: 2
      };
    } else if (archiveCaches.includes(cacheName)) {
      return {
        ...baseConfig,
        memory: false, // Disk-based storage
        healthCheckInterval: 1800000, // 30-minute health checks
        maxRetries: 1
      };
    } else {
      // Default configuration for unknown cache types
      return {
        ...baseConfig,
        memory: false, // Conservative default
        healthCheckInterval: 600000, // 10-minute health checks
        maxRetries: 2
      };
    }
  },
  /**
   * Initialize a cache instance
   * @param {string} cacheName - Cache name
   * @param {number} duration - Cache duration in milliseconds
   * @returns {Object} Cache instance
   */
  initCache(cacheName, duration = 24 * 60 * 60 * 1000) {
    if (!cacheInstances.has(cacheName)) {
      try {
        // Get cache-type-specific configuration
        const config = this.getCacheConfig(cacheName, duration);
        const cacheInstance = cache(config);
        cacheInstances.set(cacheName, cacheInstance);

        // Enable health monitoring for critical caches
        if (config.healthCheckInterval) {
          this.startHealthMonitoring(cacheName);
        }
      } catch (error) {
        consoleLog('cache', `Error initializing cache ${cacheName}: ${error.message}`, {
          action: 'INIT_ERROR',
          cacheName,
          error: error.message
        });
        // Create enhanced fallback cache object with complete API compatibility
        const fallbackCache = {
          get: (key, callback) => callback(null, null),
          getSync: (key) => null,
          put: (key, value, callback) => callback(null),
          putSync: (key, value) => true,
          delete: (key, callback) => callback(null),
          deleteSync: (key) => true,
          keys: (callback) => callback(null, []),
          keysSync: () => [],
          unlink: (callback) => callback(null),
          healthCheck: (callback) => callback(null, { healthy: false, fallback: true }),
          healthCheckSync: () => ({ healthy: false, fallback: true })
        };
        cacheInstances.set(cacheName, fallbackCache);
      }
    }
    
    // Track cache usage
    cacheUsageTracking.set(cacheName, Date.now());
    
    return cacheInstances.get(cacheName);
  },

  /**
   * Get value from cache
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @returns {Promise<any>} Cached value or null
   */
  async get(cacheName, key) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'GET', key, hit: false });
        resolve(null);
        return;
      }
      
      // Track cache usage
      cacheUsageTracking.set(cacheName, Date.now());
      
      cacheInstance.get(key, function(err, value) {
        if (err) {
          // Simplified error handling - autoRepair handles corruption automatically
          consoleLog('cache', `Error getting cache (${cacheName}): ${err.message}`, { action: 'GET', key, hit: false });
          resolve(null);
        } else {
          resolve(value);
        }
      });
    });
  },

  /**
   * Put value in cache
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @param {any} value - Value to cache
   * @returns {Promise<boolean>} Success status
   */
  async put(cacheName, key, value) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'SET', key });
        resolve(false);
        return;
      }
      
      // Track cache usage
      cacheUsageTracking.set(cacheName, Date.now());
      
      cacheInstance.put(key, value, function(err) {
        if (err) {
          consoleLog('cache', `Error putting cache (${cacheName}): ${err.message}`, { action: 'SET', key });
          resolve(false);
        } else {
          resolve(true);
        }
      });
    });
  },

  /**
   * Delete key from cache
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @returns {Promise<boolean>} Success status
   */
  async del(cacheName, key) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'DEL', key });
        resolve(false);
        return;
      }

      cacheInstance.delete(key, function(err) {
        if (err) {
          consoleLog('cache', `Error deleting cache (${cacheName}): ${err.message}`, { action: 'DEL', key });
          resolve(false);
        } else {
          resolve(true);
        }
      });
    });
  },

  /**
   * Get value from cache synchronously
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @returns {any} Cached value or null
   */
  getSync(cacheName, key) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'GET_SYNC', key, hit: false });
      return null;
    }

    // Track cache usage
    cacheUsageTracking.set(cacheName, Date.now());

    try {
      if (typeof cacheInstance.getSync === 'function') {
        return cacheInstance.getSync(key);
      } else {
        consoleLog('cache', `Sync operation not supported for cache ${cacheName}`, { action: 'GET_SYNC', key });
        return null;
      }
    } catch (error) {
      consoleLog('cache', `Error getting cache sync (${cacheName}): ${error.message}`, { action: 'GET_SYNC', key, hit: false });
      return null;
    }
  },

  /**
   * Put value in cache synchronously
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @param {any} value - Value to cache
   * @returns {boolean} Success status
   */
  putSync(cacheName, key, value) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'PUT_SYNC', key });
      return false;
    }

    // Track cache usage
    cacheUsageTracking.set(cacheName, Date.now());

    try {
      if (typeof cacheInstance.putSync === 'function') {
        cacheInstance.putSync(key, value);
        return true;
      } else {
        consoleLog('cache', `Sync operation not supported for cache ${cacheName}`, { action: 'PUT_SYNC', key });
        return false;
      }
    } catch (error) {
      consoleLog('cache', `Error putting cache sync (${cacheName}): ${error.message}`, { action: 'PUT_SYNC', key });
      return false;
    }
  },

  /**
   * Delete key from cache synchronously
   * @param {string} cacheName - Cache name
   * @param {string} key - Cache key
   * @returns {boolean} Success status
   */
  delSync(cacheName, key) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'DEL_SYNC', key });
      return false;
    }

    try {
      if (typeof cacheInstance.deleteSync === 'function') {
        cacheInstance.deleteSync(key);
        return true;
      } else {
        consoleLog('cache', `Sync operation not supported for cache ${cacheName}`, { action: 'DEL_SYNC', key });
        return false;
      }
    } catch (error) {
      consoleLog('cache', `Error deleting cache sync (${cacheName}): ${error.message}`, { action: 'DEL_SYNC', key });
      return false;
    }
  },

  /**
   * List all keys in cache
   * @param {string} cacheName - Cache name
   * @returns {Promise<Array>} Array of cache keys
   */
  async keys(cacheName) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'KEYS' });
        resolve([]);
        return;
      }

      if (typeof cacheInstance.keys === 'function') {
        cacheInstance.keys(function(err, keys) {
          if (err) {
            consoleLog('cache', `Error listing cache keys (${cacheName}): ${err.message}`, { action: 'KEYS' });
            resolve([]);
          } else {
            resolve(keys || []);
          }
        });
      } else {
        consoleLog('cache', `Keys operation not supported for cache ${cacheName}`, { action: 'KEYS' });
        resolve([]);
      }
    });
  },

  /**
   * List all keys in cache synchronously
   * @param {string} cacheName - Cache name
   * @returns {Array} Array of cache keys
   */
  keysSync(cacheName) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'KEYS_SYNC' });
      return [];
    }

    try {
      if (typeof cacheInstance.keysSync === 'function') {
        return cacheInstance.keysSync() || [];
      } else {
        consoleLog('cache', `Sync keys operation not supported for cache ${cacheName}`, { action: 'KEYS_SYNC' });
        return [];
      }
    } catch (error) {
      consoleLog('cache', `Error listing cache keys sync (${cacheName}): ${error.message}`, { action: 'KEYS_SYNC' });
      return [];
    }
  },

  /**
   * Completely remove cache directory and all its contents
   * @param {string} cacheName - Cache name
   * @returns {Promise<boolean>} Success status
   */
  async unlink(cacheName) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        consoleLog('cache', `Cache ${cacheName} not initialized`, { action: 'UNLINK' });
        resolve(false);
        return;
      }

      if (typeof cacheInstance.unlink === 'function') {
        cacheInstance.unlink(function(err) {
          if (err) {
            consoleLog('cache', `Error unlinking cache (${cacheName}): ${err.message}`, { action: 'UNLINK' });
            resolve(false);
          } else {
            // Remove from our tracking maps
            cacheInstances.delete(cacheName);
            cacheUsageTracking.delete(cacheName);
            healthMonitoring.delete(cacheName);

            consoleLog('cache', `Successfully unlinked cache ${cacheName}`, { action: 'UNLINK' });
            resolve(true);
          }
        });
      } else {
        consoleLog('cache', `Unlink operation not supported for cache ${cacheName}`, { action: 'UNLINK' });
        resolve(false);
      }
    });
  },

  /**
   * Clear entire cache by recreating the cache instance
   * @param {string} cacheName - Cache name
   * @returns {Promise<boolean>} Success status
   */
  async clear(cacheName) {
    try {
      // Remove the current cache instance
      if (cacheInstances.has(cacheName)) {
        cacheInstances.delete(cacheName);
        cacheUsageTracking.delete(cacheName);
      }
      
      // Reinitialize the cache (this effectively clears it)
      this.initCache(cacheName);
      
      consoleLog('cache', `Cache ${cacheName} cleared by recreation`, { action: 'CLEAR' });
      return true;
    } catch (error) {
      consoleLog('cache', `Error clearing cache (${cacheName}): ${error.message}`, { action: 'CLEAR' });
      return false;
    }
  },

  /**
   * Invalidate all game-related caches
   * Called when games are created, updated, or deleted
   */
  async invalidateGameCaches() {
    // Device-specific cache keys that need both mobile and desktop versions
    const deviceSpecificCacheKeys = [
      // Homepage caches
      ['homepage-games', 'discovery-games'],
      ['homepage-games', 'featured-games'],
      ['homepage-games', 'popular-tags'],
      ['homepage-games', 'top-searches'],
      ['homepage-randomized-categories', 'randomized-categories-with-games'],
      // Model caches
      ['featured-games', 'featured-games-6'],
      ['popular-tags', 'popular-tags-12'],
      ['top-searches', 'top-searches-40'],
      // Global middleware collections
      ['popular-games', 'collection'],
      ['featured-games', 'collection'],
      ['trending-games', 'collection'],
      ['recent-games', 'collection'],
      ['top-rated-games', 'collection']
    ];

    // Device-agnostic cache keys
    const regularCacheKeys = [
      ['sidebar-categories', 'random-categories'],
      ['game-stats', 'total-count'],
      ['homepage-games', 'hall-of-fame-users']
    ];

    // Invalidate device-specific caches (both mobile and desktop)
    const deviceInvalidationPromises = deviceSpecificCacheKeys.map(([cacheName, baseKey]) => 
      this.invalidateDeviceCaches(cacheName, baseKey)
    );

    // Invalidate regular caches
    const regularInvalidationPromises = regularCacheKeys.map(([cacheName, key]) => 
      this.del(cacheName, key)
    );

    await Promise.all([...deviceInvalidationPromises, ...regularInvalidationPromises]);
    consoleLog('cache', 'Invalidated all game-related caches (device-specific and regular)', { action: 'INVALIDATE_GAMES' });
  },

  /**
   * Invalidate all category-related caches
   * Called when categories are created, updated, or deleted
   */
  async invalidateCategoryCaches() {
    // Device-specific cache keys
    const deviceSpecificCacheKeys = [
      ['homepage-games', 'discovery-games'],
      ['homepage-randomized-categories', 'randomized-categories-with-games']
    ];

    // Device-agnostic cache keys
    const regularCacheKeys = [
      ['sidebar-categories', 'random-categories']
    ];

    // Invalidate device-specific caches (both mobile and desktop)
    const deviceInvalidationPromises = deviceSpecificCacheKeys.map(([cacheName, baseKey]) => 
      this.invalidateDeviceCaches(cacheName, baseKey)
    );

    // Invalidate regular caches
    const regularInvalidationPromises = regularCacheKeys.map(([cacheName, key]) => 
      this.del(cacheName, key)
    );

    await Promise.all([...deviceInvalidationPromises, ...regularInvalidationPromises]);
    consoleLog('cache', 'Invalidated all category-related caches (device-specific and regular)', { action: 'INVALIDATE_CATEGORIES' });
  },

  /**
   * Invalidate all homepage-related caches
   * Called when any homepage data changes
   */
  async invalidateHomepageCaches() {
    // Device-specific cache keys
    const deviceSpecificCacheKeys = [
      ['homepage-games', 'discovery-games'],
      ['homepage-games', 'featured-games'],
      ['homepage-games', 'popular-tags'],
      ['homepage-games', 'top-searches'],
      ['homepage-randomized-categories', 'randomized-categories-with-games']
    ];

    // Device-agnostic cache keys
    const regularCacheKeys = [
      ['homepage-games', 'hall-of-fame-users']
    ];

    // Invalidate device-specific caches (both mobile and desktop)
    const deviceInvalidationPromises = deviceSpecificCacheKeys.map(([cacheName, baseKey]) => 
      this.invalidateDeviceCaches(cacheName, baseKey)
    );

    // Invalidate regular caches
    const regularInvalidationPromises = regularCacheKeys.map(([cacheName, key]) => 
      this.del(cacheName, key)
    );

    await Promise.all([...deviceInvalidationPromises, ...regularInvalidationPromises]);
    consoleLog('cache', 'Invalidated all homepage-related caches (device-specific and regular)', { action: 'INVALIDATE_HOMEPAGE' });
  },

  /**
   * Invalidate all advertisement-related caches
   * Called when ads are created, updated, or deleted
   */
  async invalidateAdCaches() {
    try {
      // Remove the ads cache instance completely (this clears all cached ads)
      if (cacheInstances.has('ads')) {
        cacheInstances.delete('ads');
        cacheUsageTracking.delete('ads');
      }
      
      consoleLog('cache', 'Invalidated all advertisement-related caches', { action: 'INVALIDATE_ADS' });
      return true;
    } catch (error) {
      consoleLog('cache', `Error invalidating ad caches: ${error.message}`, { action: 'INVALIDATE_ADS', error: error.message });
      return false;
    }
  },

  /**
   * Invalidate all ad placement-related caches
   * Called when ad placements are created, updated, or deleted
   */
  async invalidateAdPlacementCaches() {
    try {
      // Remove the ad-placements cache instance completely (this clears all cached placements)
      if (cacheInstances.has('ad-placements')) {
        cacheInstances.delete('ad-placements');
        cacheUsageTracking.delete('ad-placements');
      }
      
      consoleLog('cache', 'Invalidated all ad placement-related caches', { action: 'INVALIDATE_AD_PLACEMENTS' });
      return true;
    } catch (error) {
      consoleLog('cache', `Error invalidating ad placement caches: ${error.message}`, { action: 'INVALIDATE_AD_PLACEMENTS', error: error.message });
      return false;
    }
  },

  /**
   * Invalidate all language-related caches
   * Called when language files are updated or system cache is cleared
   */
  async invalidateLanguageCaches() {
    try {
      // Clear both language cache instances
      if (cacheInstances.has('i18n-languages')) {
        cacheInstances.delete('i18n-languages');
        cacheUsageTracking.delete('i18n-languages');
      }
      
      if (cacheInstances.has('i18n-metadata')) {
        cacheInstances.delete('i18n-metadata');
        cacheUsageTracking.delete('i18n-metadata');
      }
      
      consoleLog('cache', 'Invalidated all language-related caches', { action: 'INVALIDATE_LANGUAGES' });
      return true;
    } catch (error) {
      consoleLog('cache', `Error invalidating language caches: ${error.message}`, { action: 'INVALIDATE_LANGUAGES', error: error.message });
      return false;
    }
  },

  /**
   * Clear all cache instances
   * Called when clearing entire system cache
   */
  async clearAllCaches() {
    try {
      // Run all the existing invalidation functions
      await this.invalidateGameCaches();
      await this.invalidateCategoryCaches();
      await this.invalidateHomepageCaches();
      await this.invalidateAdCaches();
      await this.invalidateAdPlacementCaches();
      await this.invalidateLanguageCaches();
      
      consoleLog('cache', 'Successfully cleared all system caches', { action: 'CLEAR_ALL' });
      return true;
    } catch (error) {
      consoleLog('cache', `Error clearing all caches: ${error.message}`, { action: 'CLEAR_ALL', error: error.message });
      return false;
    }
  },

  /**
   * Clean up unused cache instances
   * Removes cache instances that haven't been used in the specified time
   * @param {number} maxAge - Maximum age in milliseconds (default: 24 hours)
   */
  async cleanupUnusedCaches(maxAge = 24 * 60 * 60 * 1000) {
    const now = Date.now();
    const unusedCaches = [];
    
    // Find unused caches
    cacheUsageTracking.forEach((lastUsed, cacheName) => {
      if (now - lastUsed > maxAge) {
        unusedCaches.push(cacheName);
      }
    });
    
    // Remove unused cache instances
    for (const cacheName of unusedCaches) {
      cacheInstances.delete(cacheName);
      cacheUsageTracking.delete(cacheName);
    }
    
    if (unusedCaches.length > 0) {
      consoleLog('cache', 'Cleaned up unused cache instances', {
        action: 'CLEANUP_UNUSED',
        removedCaches: unusedCaches,
        remainingCaches: cacheInstances.size
      });
    }
    
    return unusedCaches.length;
  },

  /**
   * Get comprehensive cache statistics for monitoring and debugging
   * @returns {Object} Detailed cache statistics
   */
  getCacheStats() {
    const cacheNames = Array.from(cacheInstances.keys());
    const stats = {
      // Basic statistics
      totalCacheInstances: cacheInstances.size,
      cacheNames: cacheNames,
      lastUsageTimes: Object.fromEntries(cacheUsageTracking),

      // Health monitoring status
      healthMonitoring: Object.fromEntries(healthMonitoring),
      totalMonitoredCaches: Array.from(healthMonitoring.values()).filter(status => status).length,

      // Cache configuration breakdown
      cacheConfigurations: {},
      performanceTiers: {
        highPerformance: [],
        standard: [],
        archive: [],
        unknown: []
      },

      // Usage statistics
      usageStatistics: {
        recentlyUsed: 0,
        oldCaches: 0,
        averageAge: 0
      }
    };

    // Analyze each cache configuration and performance tier
    const now = Date.now();
    let totalAge = 0;
    const oneHour = 60 * 60 * 1000;

    cacheNames.forEach(cacheName => {
      // Get configuration for this cache type
      const config = this.getCacheConfig(cacheName, 24 * 60 * 60 * 1000);
      stats.cacheConfigurations[cacheName] = {
        memory: config.memory,
        persist: config.persist,
        autoRepair: config.autoRepair,
        healthCheckInterval: config.healthCheckInterval,
        maxRetries: config.maxRetries
      };

      // Categorize by performance tier
      if (config.memory && config.healthCheckInterval === 300000) {
        stats.performanceTiers.highPerformance.push(cacheName);
      } else if (!config.memory && config.healthCheckInterval === 600000) {
        stats.performanceTiers.standard.push(cacheName);
      } else if (!config.memory && config.healthCheckInterval === 1800000) {
        stats.performanceTiers.archive.push(cacheName);
      } else {
        stats.performanceTiers.unknown.push(cacheName);
      }

      // Calculate usage statistics
      const lastUsed = cacheUsageTracking.get(cacheName) || now;
      const age = now - lastUsed;
      totalAge += age;

      if (age < oneHour) {
        stats.usageStatistics.recentlyUsed++;
      } else {
        stats.usageStatistics.oldCaches++;
      }
    });

    // Calculate average age
    if (cacheNames.length > 0) {
      stats.usageStatistics.averageAge = Math.round(totalAge / cacheNames.length / 1000 / 60); // in minutes
    }

    return stats;
  },

  /**
   * Force cleanup of specific cache instance
   * @param {string} cacheName - Cache name to remove
   */
  async removeCacheInstance(cacheName) {
    if (cacheInstances.has(cacheName)) {
      cacheInstances.delete(cacheName);
      cacheUsageTracking.delete(cacheName);
      
      consoleLog('cache', 'Removed cache instance', {
        action: 'REMOVE_INSTANCE',
        cacheName,
        remainingCaches: cacheInstances.size
      });
      
      return true;
    }
    
    return false;
  },

  /**
   * Clean up corrupted cache files by clearing and reinitializing
   * @param {string} cacheName - Cache name to clean up
   * @returns {Promise<boolean>} Success status
   */
  async cleanupCorruptedCache(cacheName) {
    try {
      // Remove the corrupted cache instance
      if (cacheInstances.has(cacheName)) {
        cacheInstances.delete(cacheName);
        cacheUsageTracking.delete(cacheName);
      }
      
      // Reinitialize with a fresh instance
      this.initCache(cacheName);
      
      consoleLog('cache', `Cleaned up corrupted cache: ${cacheName}`, {
        action: 'CLEANUP_CORRUPTED',
        cacheName
      });
      
      return true;
    } catch (error) {
      consoleLog('cache', `Error cleaning up corrupted cache ${cacheName}: ${error.message}`, {
        action: 'CLEANUP_CORRUPTED_ERROR',
        cacheName,
        error: error.message
      });
      return false;
    }
  },

  /**
   * Check health of individual cache instance
   * @param {string} cacheName - Cache name to check
   * @returns {Promise<Object>} Health status object
   */
  async healthCheck(cacheName) {
    return new Promise((resolve) => {
      const cacheInstance = cacheInstances.get(cacheName);
      if (!cacheInstance) {
        resolve({ healthy: false, error: 'Cache not initialized', cacheName });
        return;
      }

      if (typeof cacheInstance.healthCheck === 'function') {
        cacheInstance.healthCheck((err, status) => {
          if (err) {
            resolve({ healthy: false, error: err.message, cacheName });
          } else {
            resolve({ healthy: true, ...status, cacheName });
          }
        });
      } else {
        // Fallback cache or cache without health check
        resolve({ healthy: false, fallback: true, cacheName });
      }
    });
  },

  /**
   * Start health monitoring for a cache instance
   * @param {string} cacheName - Cache name to monitor
   * @returns {boolean} Success status
   */
  startHealthMonitoring(cacheName) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      consoleLog('cache', `Cannot start health monitoring: Cache ${cacheName} not initialized`, {
        action: 'START_HEALTH_MONITORING_ERROR',
        cacheName
      });
      return false;
    }

    if (typeof cacheInstance.startHealthMonitoring === 'function') {
      try {
        cacheInstance.startHealthMonitoring((healthReport) => {
          consoleLog('cache', `Health monitoring report for ${cacheName}`, {
            action: 'HEALTH_MONITORING_REPORT',
            cacheName,
            ...healthReport
          });
        });
        healthMonitoring.set(cacheName, true);
        consoleLog('cache', `Started health monitoring for ${cacheName}`, {
          action: 'START_HEALTH_MONITORING',
          cacheName
        });
        return true;
      } catch (error) {
        consoleLog('cache', `Error starting health monitoring for ${cacheName}: ${error.message}`, {
          action: 'START_HEALTH_MONITORING_ERROR',
          cacheName,
          error: error.message
        });
        return false;
      }
    }

    return false;
  },

  /**
   * Stop health monitoring for a cache instance
   * @param {string} cacheName - Cache name to stop monitoring
   * @returns {boolean} Success status
   */
  stopHealthMonitoring(cacheName) {
    const cacheInstance = cacheInstances.get(cacheName);
    if (!cacheInstance) {
      return false;
    }

    if (typeof cacheInstance.stopHealthMonitoring === 'function') {
      try {
        cacheInstance.stopHealthMonitoring();
        healthMonitoring.set(cacheName, false);
        consoleLog('cache', `Stopped health monitoring for ${cacheName}`, {
          action: 'STOP_HEALTH_MONITORING',
          cacheName
        });
        return true;
      } catch (error) {
        consoleLog('cache', `Error stopping health monitoring for ${cacheName}: ${error.message}`, {
          action: 'STOP_HEALTH_MONITORING_ERROR',
          cacheName,
          error: error.message
        });
        return false;
      }
    }

    return false;
  },

  /**
   * Get system-wide health status for all cache instances
   * @returns {Promise<Object>} System health report
   */
  async getHealthStatus() {
    const cacheNames = Array.from(cacheInstances.keys());
    const healthChecks = cacheNames.map(cacheName => this.healthCheck(cacheName));
    const results = await Promise.all(healthChecks);

    const healthReport = {
      timestamp: new Date().toISOString(),
      totalCaches: cacheNames.length,
      healthyCaches: results.filter(r => r.healthy).length,
      unhealthyCaches: results.filter(r => !r.healthy).length,
      monitoring: Object.fromEntries(healthMonitoring),
      details: results
    };

    return healthReport;
  },

  /**
   * Generate device-specific cache key
   * @param {string} baseKey - Base cache key
   * @param {boolean} isMobile - Whether the request is from mobile device
   * @returns {string} Device-specific cache key
   */
  getDeviceCacheKey(baseKey, isMobile) {
    const deviceSuffix = isMobile ? '-mobile' : '-desktop';
    return `${baseKey}${deviceSuffix}`;
  },

  /**
   * Invalidate both mobile and desktop versions of a cache key
   * @param {string} cacheName - Cache name
   * @param {string} baseKey - Base cache key (without device suffix)
   * @returns {Promise<boolean>} Success status
   */
  async invalidateDeviceCaches(cacheName, baseKey) {
    try {
      const mobileKey = this.getDeviceCacheKey(baseKey, true);
      const desktopKey = this.getDeviceCacheKey(baseKey, false);
      
      await Promise.all([
        this.del(cacheName, mobileKey),
        this.del(cacheName, desktopKey)
      ]);
      
      consoleLog('cache', `Invalidated device-specific caches for ${baseKey}`, {
        action: 'INVALIDATE_DEVICE_CACHES',
        cacheName,
        baseKey,
        mobileKey,
        desktopKey
      });
      
      return true;
    } catch (error) {
      consoleLog('cache', `Error invalidating device caches: ${error.message}`, {
        action: 'INVALIDATE_DEVICE_CACHES_ERROR',
        cacheName,
        baseKey,
        error: error.message
      });
      return false;
    }
  }
};

export default CacheUtils;