/**
 * Device Detection Utilities
 * 
 * Lightweight mobile device detection based on User-Agent strings.
 * Used to filter games for mobile compatibility.
 */

/**
 * Detect if the request is from a mobile device
 * @param {Object} req - Express request object
 * @returns {boolean} - True if mobile/tablet device
 */
export const isMobileDevice = (req) => {
    const ua = req.get('User-Agent') || '';
    
    // Check for mobile devices and tablets
    const mobilePatterns = [
        /Mobile/i,
        /Android/i,
        /iPhone/i,
        /iPad/i,
        /iPod/i,
        /BlackBerry/i,
        /Windows Phone/i,
        /webOS/i,
        /Opera Mini/i
    ];
    
    return mobilePatterns.some(pattern => pattern.test(ua));
};

/**
 * Get device type as string
 * @param {Object} req - Express request object
 * @returns {string} - 'mobile' or 'desktop'
 */
export const getDeviceType = (req) => {
    return isMobileDevice(req) ? 'mobile' : 'desktop';
};

/**
 * Check if device supports touch
 * @param {Object} req - Express request object
 * @returns {boolean} - True if touch-enabled device
 */
export const isTouchDevice = (req) => {
    // Mobile devices typically support touch
    return isMobileDevice(req);
};