import fs from 'fs-extra';
import path from 'path';
import { fileURLToPath } from 'url';
import { consoleLog } from './logger.js';
import { getSetting } from '../models/settings.js';
import CacheUtils from './cache.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const rootDir = path.resolve(__dirname, '../..');

class I18n {
    constructor() {
        this.languagesDir = path.join(rootDir, 'languages');
        this.fallbackLanguage = 'en'; // Fallback if database is not available
        this.defaultLanguage = 'en'; // Add defaultLanguage property for consistency
        this.languageCache = CacheUtils.initCache('i18n-languages', 6 * 60 * 60 * 1000); // 6 hours
        this.metadataCache = CacheUtils.initCache('i18n-metadata', 24 * 60 * 60 * 1000); // 24 hours
    }

    /**
     * Get default language from database settings
     * @returns {Promise<string>} Default language code
     */
    async getDefaultLanguage() {
        try {
            const defaultLang = await getSetting('default_language');
            return defaultLang || this.fallbackLanguage;
        } catch (error) {
            consoleLog('warn', 'Failed to get default language from settings, using fallback', { error });
            return this.fallbackLanguage;
        }
    }

    /**
     * Load language file with caching
     * @param {string} languageCode 
     * @returns {Object} Language data
     */
    async loadLanguage(languageCode) {
        try {
            // Check cache first
            const cacheKey = `lang-${languageCode}`;
            const cachedData = await CacheUtils.get('i18n-languages', cacheKey);

            // Enhanced cache validation - check for both null and undefined
            if (cachedData !== null && cachedData !== undefined) {
                return cachedData;
            }

            // Load from disk if not cached
            const languageFile = path.join(this.languagesDir, `${languageCode}.json`);

            if (!await fs.pathExists(languageFile)) {
                // Cache empty object for non-existent files
                const emptyLangData = {};
                await CacheUtils.put('i18n-languages', cacheKey, emptyLangData);
                return emptyLangData;
            }

            const fileContent = await fs.readFile(languageFile, 'utf8');
            const languageData = JSON.parse(fileContent);

            // Ensure languageData is an object
            const validLanguageData = languageData && typeof languageData === 'object' ? languageData : {};

            // Cache the loaded data
            await CacheUtils.put('i18n-languages', cacheKey, validLanguageData);

            return validLanguageData;
        } catch (error) {
            consoleLog('error', `Failed to load language file ${languageCode}`, { error });
            return {};
        }
    }

    /**
     * Get available languages with caching
     * @returns {Array} List of available language codes
     */
    async getAvailableLanguages() {
        try {
            // Check cache first
            const cacheKey = 'available-languages';
            const cachedLanguages = await CacheUtils.get('i18n-metadata', cacheKey);

            // Enhanced cache validation - check for both null and undefined
            if (cachedLanguages !== null && cachedLanguages !== undefined && Array.isArray(cachedLanguages)) {
                return cachedLanguages;
            }

            // Load from disk if not cached
            if (!await fs.pathExists(this.languagesDir)) {
                consoleLog('warn', 'Languages directory does not exist, using fallback', { dir: this.languagesDir });
                return [this.fallbackLanguage];
            }

            const files = await fs.readdir(this.languagesDir);
            const languages = files
                .filter(file => file.endsWith('.json'))
                .map(file => file.replace('.json', ''));

            // Ensure we always have at least the fallback language
            if (languages.length === 0) {
                languages.push(this.fallbackLanguage);
            }

            // Cache the result
            await CacheUtils.put('i18n-metadata', cacheKey, languages);

            return languages;
        } catch (error) {
            consoleLog('error', 'Failed to get available languages', { error });
            return [this.fallbackLanguage];
        }
    }

    /**
     * Check if language is RTL with caching
     * @param {string} languageCode 
     * @returns {boolean}
     */
    async isRTL(languageCode) {
        try {
            // Check cache first
            const cacheKey = `rtl-${languageCode}`;
            const cachedRTL = await CacheUtils.get('i18n-metadata', cacheKey);

            // Enhanced cache validation - check for both null and undefined
            if (cachedRTL !== null && cachedRTL !== undefined) {
                return Boolean(cachedRTL);
            }

            // Load from language data if not cached
            const languageData = await this.loadLanguage(languageCode);
            const isRTL = Boolean(languageData._meta?.isRTL || false);

            // Cache the result
            await CacheUtils.put('i18n-metadata', cacheKey, isRTL);

            return isRTL;
        } catch (error) {
            consoleLog('error', `Failed to check RTL for language ${languageCode}`, { error });
            return false;
        }
    }

    /**
     * Get nested property value from object using dot notation
     * @param {Object} obj 
     * @param {string} path 
     * @returns {*}
     */
    getNestedValue(obj, path) {
        return path.split('.').reduce((current, key) => {
            return current && current[key] !== undefined ? current[key] : undefined;
        }, obj);
    }

    /**
     * Replace variables in translation string
     * @param {string} text 
     * @param {Object} variables 
     * @returns {string}
     */
    interpolate(text, variables = {}) {
        if (!text || typeof text !== 'string') return text;

        return text.replace(/\{(\w+)\}/g, (match, key) => {
            return variables[key] !== undefined ? variables[key] : match;
        });
    }

    /**
     * Translate a key
     * @param {string} key Translation key (e.g. 'nav.profile')
     * @param {Object} variables Variables for interpolation
     * @param {string} languageCode Language code
     * @returns {Promise<string>} Translated string
     */
    async translate(key, variables = {}, languageCode = null) {
        try {
            // Get default language from database if not provided
            if (!languageCode) {
                languageCode = await this.getDefaultLanguage();
            }
            
            // Load target language fresh every time
            const languageData = await this.loadLanguage(languageCode);
            let translation = this.getNestedValue(languageData, key);

            // Fallback to default language if not found
            const defaultLanguage = await this.getDefaultLanguage();
            if (translation === undefined && languageCode !== defaultLanguage) {
                const defaultData = await this.loadLanguage(defaultLanguage);
                translation = this.getNestedValue(defaultData, key);
            }

            // Final fallback to key itself
            if (translation === undefined) {
                translation = key;
            }

            // Interpolate variables
            return this.interpolate(translation, variables);
        } catch (error) {
            consoleLog('error', `Translation error for key "${key}"`, { error });
            return key; // Return key as fallback
        }
    }

    /**
     * Synchronous translation for template use
     * @param {string} key Translation key
     * @param {Object} variables Variables for interpolation
     * @param {string} languageCode Language code
     * @returns {string} Translated string
     */
    translateSync(key, variables = {}, languageCode = null) {
        try {
            // Use fallback language if no language code provided (can't make async DB call in sync method)
            if (!languageCode) {
                languageCode = this.fallbackLanguage;
            }
            
            // Load directly from file system synchronously using ES6 imports
            const languageFilePath = path.join(this.languagesDir, `${languageCode}.json`);
            
            let languageData = {};
            if (fs.pathExistsSync(languageFilePath)) {
                const fileContent = fs.readFileSync(languageFilePath, 'utf8');
                languageData = JSON.parse(fileContent);
            }
            
            let translation = this.getNestedValue(languageData, key);
            
            // Fallback to default language if not found
            if (translation === undefined && languageCode !== this.fallbackLanguage) {
                const defaultFilePath = path.join(this.languagesDir, `${this.fallbackLanguage}.json`);
                
                if (fs.pathExistsSync(defaultFilePath)) {
                    const fileContent = fs.readFileSync(defaultFilePath, 'utf8');
                    const defaultData = JSON.parse(fileContent);
                    translation = this.getNestedValue(defaultData, key);
                }
            }
            
            // Final fallback to key itself
            if (translation === undefined) {
                translation = key;
            }
            
            // Interpolate variables
            return this.interpolate(translation, variables);
        } catch (error) {
            consoleLog('error', `Sync translation error for key "${key}"`, { error });
            return key; // Return key as fallback
        }
    }

    /**
     * Translate with pluralization support
     * @param {string} key Translation key
     * @param {number} count Count for pluralization
     * @param {Object} variables Variables for interpolation
     * @param {string} languageCode Language code
     * @returns {Promise<string>} Translated string
     */
    async translatePlural(key, count, variables = {}, languageCode = null) {
        // Use provided language code or get default
        const targetLanguageCode = languageCode || await this.getDefaultLanguage();
        const pluralKey = count === 1 ? key : `${key}_plural`;
        const mergedVariables = { count, ...variables };
        return this.translate(pluralKey, mergedVariables, targetLanguageCode);
    }

    /**
     * Get language metadata with caching
     * @param {string} languageCode 
     * @returns {Promise<Object>}
     */
    async getLanguageInfo(languageCode) {
        try {
            // Input validation - handle edge cases
            if (!languageCode || typeof languageCode !== 'string') {
                return this.getDefaultLanguageInfo(this.fallbackLanguage);
            }

            // Check cache first
            const cacheKey = `info-${languageCode}`;
            const cachedInfo = await CacheUtils.get('i18n-metadata', cacheKey);

            // Enhanced cache validation using our validation method
            if (cachedInfo !== null && cachedInfo !== undefined && this.isValidLanguageInfo(cachedInfo)) {
                return cachedInfo;
            }

            // Load from language data if not cached or invalid
            const languageData = await this.loadLanguage(languageCode);
            const meta = languageData._meta || {};

            const languageInfo = {
                code: languageCode,
                name: meta.name || languageCode,
                nativeName: meta.nativeName || languageCode,
                isRTL: Boolean(meta.isRTL || false),
                flag: meta.flag || '🌐'
            };

            // Cache the result
            await CacheUtils.put('i18n-metadata', cacheKey, languageInfo);

            return languageInfo;
        } catch (error) {
            consoleLog('error', `Failed to get language info for ${languageCode}`, { error });
            return this.getDefaultLanguageInfo(languageCode);
        }
    }

    /**
     * Get all translations for a language (for client-side)
     * @param {string} languageCode
     * @returns {Object}
     */
    async getAllTranslations(languageCode) {
        return await this.loadLanguage(languageCode);
    }

    /**
     * Validate language info object structure
     * @param {Object} languageInfo - Language info object to validate
     * @returns {boolean} True if valid structure
     */
    isValidLanguageInfo(languageInfo) {
        if (!languageInfo || typeof languageInfo !== 'object') {
            return false;
        }

        const requiredProps = ['code', 'name', 'nativeName', 'isRTL'];
        for (const prop of requiredProps) {
            if (!(prop in languageInfo)) {
                return false;
            }
        }

        // Type validation
        if (typeof languageInfo.code !== 'string' ||
            typeof languageInfo.name !== 'string' ||
            typeof languageInfo.nativeName !== 'string' ||
            typeof languageInfo.isRTL !== 'boolean') {
            return false;
        }

        return true;
    }

    /**
     * Get default language info structure for fallback
     * @param {string} languageCode - Language code to use
     * @returns {Object} Default language info
     */
    getDefaultLanguageInfo(languageCode = null) {
        const code = languageCode || this.fallbackLanguage;
        return {
            code: code,
            name: code === 'en' ? 'English' : code,
            nativeName: code === 'en' ? 'English' : code,
            isRTL: false,
            flag: '🌐'
        };
    }

    /**
     * Clear language caches (for cache invalidation)
     * @param {string} languageCode - Specific language to clear, or null for all
     */
    async clearLanguageCache(languageCode = null) {
        try {
            if (languageCode) {
                // Clear specific language cache
                await CacheUtils.del('i18n-languages', `lang-${languageCode}`);
                await CacheUtils.del('i18n-metadata', `rtl-${languageCode}`);
                await CacheUtils.del('i18n-metadata', `info-${languageCode}`);
                consoleLog('info', `Cleared cache for language: ${languageCode}`);
            } else {
                // Clear all language caches
                await CacheUtils.clear('i18n-languages');
                await CacheUtils.clear('i18n-metadata');
                consoleLog('info', 'Cleared all language caches');
            }
        } catch (error) {
            consoleLog('error', 'Failed to clear language cache', { error, languageCode });
        }
    }
}

// Create singleton instance
const i18n = new I18n();

/**
 * Global translation function
 * @param {string} key Translation key
 * @param {Object} variables Variables for interpolation
 * @param {string} languageCode Language code
 * @returns {Promise<string>} Translated string
 */
export async function __(key, variables = {}, languageCode = 'en') {
    return await i18n.translate(key, variables, languageCode);
}

/**
 * Global pluralization function
 * @param {string} key Translation key
 * @param {number} count Count for pluralization
 * @param {Object} variables Variables for interpolation
 * @param {string} languageCode Language code
 * @returns {Promise<string>} Translated string
 */
export async function __n(key, count, variables = {}, languageCode = 'en') {
    return await i18n.translatePlural(key, count, variables, languageCode);
}

export default i18n;