import axios from 'axios';
import fs from 'fs';
import path from 'path';
import unzipper from 'unzipper';
import { detectRomSystem, detectRomSystemFromZipBuffer } from './rom.js';
import ImageProcessor from './image_processor.js';
import { getCategoryByName } from '../models/categories.js';
import { consoleLog } from './logger.js';
import { create } from '../models/crud.js';
import { generateUniqueSlug } from './sanitize.js';
import CacheUtils from './cache.js';

// Cache for available icons to avoid repeated filesystem calls
let availableIconsCache = null;

/**
 * Detects if buffer contains SWF file by checking magic bytes
 * @param {Buffer} buffer - File buffer to analyze
 * @returns {boolean} - True if SWF file detected
 */
function isSWFFile(buffer) {
    if (buffer.length < 3) return false;

    // SWF magic bytes: "FWS" (uncompressed), "CWS" (zlib), "ZWS" (lzma)
    const signature = buffer.subarray(0, 3).toString('ascii');
    return ['FWS', 'CWS', 'ZWS'].includes(signature);
}

/**
 * Detects if buffer contains ZIP file by checking magic bytes
 * @param {Buffer} buffer - File buffer to analyze
 * @returns {boolean} - True if ZIP file detected
 */
function isZIPFile(buffer) {
    if (buffer.length < 4) return false;

    // ZIP magic bytes: 0x504B0304 (PK..) or 0x504B0506 (PK..) or 0x504B0708 (PK..)
    const signature = buffer.readUInt32LE(0);
    return [0x04034b50, 0x06054b50, 0x08074b50].includes(signature);
}

/**
 * Comprehensive SWF file validation
 * @param {Buffer} buffer - SWF file buffer
 * @param {string} gameSlug - Game identifier for logging
 * @returns {Object} - Validation result with details
 */
function validateSWFFile(buffer, gameSlug) {
    const validation = {
        isValid: false,
        signature: null,
        version: null,
        fileSize: null,
        errors: []
    };

    try {
        // Check minimum file size
        if (buffer.length < 8) {
            validation.errors.push('SWF file too small (minimum 8 bytes)');
            return validation;
        }

        // Check signature
        const signature = buffer.subarray(0, 3).toString('ascii');
        if (!['FWS', 'CWS', 'ZWS'].includes(signature)) {
            validation.errors.push(`Invalid SWF signature: ${signature}`);
            return validation;
        }
        validation.signature = signature;

        // Check version
        const version = buffer.readUInt8(3);
        if (version < 1 || version > 47) { // SWF versions 1-47 are valid
            validation.errors.push(`Invalid SWF version: ${version}`);
        }
        validation.version = version;

        // Check declared file size
        const declaredSize = buffer.readUInt32LE(4);
        validation.fileSize = declaredSize;

        if (declaredSize < 8) {
            validation.errors.push(`Invalid declared file size: ${declaredSize}`);
        }

        // Size warning (not error) if mismatch
        if (declaredSize !== buffer.length) {
            consoleLog('warn', 'SWF declared size mismatch', {
                gameSlug,
                declaredSize,
                actualSize: buffer.length
            });
        }

        validation.isValid = validation.errors.length === 0;

        consoleLog('info', 'SWF validation completed', {
            gameSlug,
            isValid: validation.isValid,
            signature: validation.signature,
            version: validation.version,
            declaredSize: validation.fileSize,
            actualSize: buffer.length,
            errors: validation.errors
        });

    } catch (error) {
        validation.errors.push(`Validation error: ${error.message}`);
        consoleLog('error', 'SWF validation failed', {
            gameSlug,
            error: error.message
        });
    }

    return validation;
}

/**
 * Ensures Flash games directory exists with proper permissions
 * @param {string} gameUploadsDir - Base uploads directory
 * @returns {string} - Flash games directory path
 */
function ensureFlashDirectory(gameUploadsDir) {
    const flashDir = path.join(gameUploadsDir, 'flash');

    if (!fs.existsSync(flashDir)) {
        try {
            fs.mkdirSync(flashDir, { recursive: true, mode: 0o755 });
            consoleLog('info', 'Created Flash games directory', { flashDir });
        } catch (error) {
            consoleLog('error', 'Failed to create Flash directory', {
                flashDir,
                error: error.message
            });
            throw new Error(`Failed to create Flash games directory: ${error.message}`);
        }
    }

    // Verify directory is writable
    try {
        fs.accessSync(flashDir, fs.constants.W_OK);
    } catch (error) {
        throw new Error(`Flash games directory is not writable: ${flashDir}`);
    }

    return flashDir;
}

/**
 * Gets list of available heroicons from the filesystem
 * @returns {Promise<string[]>} Array of available icon names (without .svg extension)
 */
async function getAvailableHeroicons() {
    if (availableIconsCache !== null) {
        return availableIconsCache;
    }
    
    try {
        const heroiconsDir = path.join(process.cwd(), 'public', 'assets', 'images', 'heroicons');
        
        if (!fs.existsSync(heroiconsDir)) {
            consoleLog('warn', 'Heroicons directory not found', { path: heroiconsDir });
            availableIconsCache = [];
            return availableIconsCache;
        }
        
        const files = fs.readdirSync(heroiconsDir);
        const iconNames = files
            .filter(file => file.endsWith('.svg'))
            .map(file => file.replace('.svg', ''));
        
        availableIconsCache = iconNames;
        consoleLog('info', 'Loaded available heroicons', { count: iconNames.length });
        
        return availableIconsCache;
    } catch (error) {
        consoleLog('error', 'Failed to load available heroicons', { error: error.message });
        availableIconsCache = [];
        return availableIconsCache;
    }
}

/**
 * Checks if a specific heroicon exists
 * @param {string} iconName - Icon name (without .svg extension)
 * @returns {Promise<boolean>} Whether the icon exists
 */
async function isHeroiconAvailable(iconName) {
    if (!iconName) return false;
    
    const availableIcons = await getAvailableHeroicons();
    return availableIcons.includes(iconName);
}

/**
 * Gets a valid heroicon name, falling back to default if the requested one doesn't exist
 * @param {string} requestedIcon - The requested icon name
 * @param {string} defaultIcon - Default icon to use if requested doesn't exist (default: 'squares-2x2')
 * @returns {Promise<string>} Valid icon name
 */
async function getValidHeroicon(requestedIcon, defaultIcon = 'squares-2x2') {
    if (!requestedIcon) {
        return defaultIcon;
    }
    
    const iconExists = await isHeroiconAvailable(requestedIcon);
    
    if (iconExists) {
        return requestedIcon;
    }
    
    // Check if default icon exists, if not use a guaranteed fallback
    const defaultExists = await isHeroiconAvailable(defaultIcon);
    if (defaultExists) {
        consoleLog('warn', 'Requested icon not found, using default', { 
            requestedIcon, 
            defaultIcon 
        });
        return defaultIcon;
    }
    
    // Last resort - use 'heart' as it's a common icon
    const heartExists = await isHeroiconAvailable('heart');
    if (heartExists) {
        consoleLog('warn', 'Default icon not found, using heart', { 
            requestedIcon, 
            defaultIcon 
        });
        return 'heart';
    }
    
    // If even heart doesn't exist, just return the default and let the UI handle it
    consoleLog('error', 'No valid heroicon found, returning default anyway', { 
        requestedIcon, 
        defaultIcon 
    });
    return defaultIcon;
}

/**
 * Gets appropriate heroicon based on category name
 * @param {string} categoryName - Category name to get icon for
 * @returns {Promise<string>} Appropriate icon name
 */
async function getIconForCategory(categoryName) {
    if (!categoryName) {
        return await getValidHeroicon('squares-2x2');
    }
    
    // Category to icon mapping
    const categoryIconMap = {
        // Game genres
        'action': 'bolt',
        'adventure': 'map',
        'puzzle': 'puzzle-piece',
        'racing': 'truck',
        'driving': 'truck',
        'sports': 'trophy',
        'fighting': 'hand-raised',
        'shooting': 'viewfinder-circle',
        'rpg': 'user-group',
        'strategy': 'squares-2x2',
        'simulation': 'computer-desktop',
        'arcade': 'rocket-launch',
        'platform': 'arrow-up',
        'platformer': 'arrow-up',
        'educational': 'academic-cap',
        'music': 'musical-note',
        'card': 'rectangle-group',
        'board': 'table-cells',
        'trivia': 'question-mark-circle',
        'word': 'chat-bubble-left',
        'math': 'calculator',
        'memory': 'light-bulb',
        'skill': 'star',
        'casual': 'heart',
        'multiplayer': 'users',
        'horror': 'exclamation-triangle',
        'sci-fi': 'rocket-launch',
        'fantasy': 'sparkles',
        'retro': 'tv',
        'classic': 'squares-2x2',
        // Common variations
        'action-adventure': 'bolt',
        'role-playing': 'user-group',
        'first-person-shooter': 'viewfinder-circle',
        'real-time-strategy': 'squares-2x2'
    };
    
    // Try to find a match by category name (case insensitive)
    const categoryKey = categoryName.toLowerCase().replace(/[^a-z0-9]/g, '');
    let iconName = null;
    
    // Direct match
    if (categoryIconMap[categoryKey]) {
        iconName = categoryIconMap[categoryKey];
    } else {
        // Partial match
        for (const [key, value] of Object.entries(categoryIconMap)) {
            if (categoryKey.includes(key) || key.includes(categoryKey)) {
                iconName = value;
                break;
            }
        }
    }
    
    // If no mapping found, use default
    if (!iconName) {
        iconName = 'squares-2x2';
    }
    
    // Validate the icon exists
    return await getValidHeroicon(iconName);
}

/**
 * Maps API category to local category ID, creating the category if it doesn't exist
 * @param {string} apiCategory - Category from API
 * @returns {Promise<number>} Local category ID
 */
export async function mapApiCategoryToLocal(apiCategory) {
    if (!apiCategory) return 1; // Default category
    
    // Create a mapping object for common categories
    const categoryMap = {
        'Action': 'Action',
        'Adventure': 'Adventure', 
        'RPG': 'RPG',
        'Strategy': 'Strategy',
        'Sports': 'Sports',
        'Platform': 'Platform',
        'Puzzle': 'Puzzle',
        'Racing': 'Racing',
        'Shooter': 'Shooter',
        'Simulation': 'Simulation',
        'Fighting': 'Fighting',
        'Action-Adventure': 'Action',
        'Action Adventure': 'Action'
    };
    
    const mappedCategory = categoryMap[apiCategory] || apiCategory;
    
    try {
        // Try to find exact match using model
        let category = await getCategoryByName(mappedCategory);
        if (category) {
            return category.id;
        }
        
        // If no exact match, try to find by original API category name
        if (mappedCategory !== apiCategory) {
            category = await getCategoryByName(apiCategory);
            if (category) {
                return category.id;
            }
        }
        
        // If category doesn't exist, create it
        const newCategoryId = await createCategory(mappedCategory);
        if (newCategoryId) {
            consoleLog('info', 'Created new category during import', { 
                categoryName: mappedCategory, 
                categoryId: newCategoryId,
                originalApiCategory: apiCategory
            });
            return newCategoryId;
        }
        
        // Default to first category if creation fails
        return 1;
    } catch (error) {
        consoleLog('error', 'Category mapping error', { error: error.message });
        return 1; // Default category
    }
}

/**
 * Creates a new category with default values
 * @param {string} categoryName - Name of the category to create
 * @returns {Promise<number|null>} Created category ID or null if failed
 */
async function createCategory(categoryName) {
    try {
        // Generate unique slug for the category
        const categorySlug = await generateUniqueSlug(categoryName, 'categories');
        
        // Default colors for auto-created categories
        const defaultColors = [
            '#3B82F6', // Blue
            '#10B981', // Green
            '#F59E0B', // Yellow
            '#EF4444', // Red
            '#8B5CF6', // Purple
            '#06B6D4', // Cyan
            '#F97316', // Orange
            '#84CC16', // Lime
            '#EC4899', // Pink
            '#6B7280'  // Gray
        ];
        
        // Get a random color from the default colors
        const randomColor = defaultColors[Math.floor(Math.random() * defaultColors.length)];
        
        // Get appropriate icon for the category
        const iconName = await getIconForCategory(categoryName);
        
        // Create category with default values
        const categoryData = {
            name: categoryName,
            slug: categorySlug,
            description: `Auto-created category for ${categoryName} games`,
            color: randomColor,
            icon: iconName,
            is_active: 1,
            sort_order: 0
        };
        
        const categoryId = await create('categories', categoryData);
        
        if (categoryId) {
            // Invalidate category-related caches
            await CacheUtils.invalidateCache('sidebar-categories');
            await CacheUtils.invalidateCache('random-categories');
            
            consoleLog('info', 'Successfully created new category', {
                categoryName,
                categoryId,
                slug: categorySlug,
                color: randomColor,
                icon: iconName
            });
            return categoryId;
        }
        
        return null;
    } catch (error) {
        consoleLog('error', 'Failed to create category', {
            categoryName,
            error: error.message
        });
        return null;
    }
}

/**
 * Downloads and processes thumbnail from API
 * @param {string} thumbnailUrl - URL of thumbnail
 * @param {string} gameSlug - Game slug for filename
 * @returns {Promise<string>} Processed thumbnail data
 */
export async function downloadAndProcessThumbnail(thumbnailUrl, gameSlug) {
    try {
        const response = await axios.get(thumbnailUrl, {
            responseType: 'arraybuffer',
            timeout: 30000 // 30 seconds
        });
        
        const buffer = Buffer.from(response.data);
        const uploadsDir = path.join(process.cwd(), 'uploads', 'images', 'games');
        
        // Use existing image processor
        const processedImages = await ImageProcessor.processImage(
            buffer,
            `${gameSlug}_thumbnail.jpg`,
            'games',
            uploadsDir,
            gameSlug
        );
        
        // Return JSON string for database storage
        return JSON.stringify({
            webp: processedImages.webp,
            original: processedImages.original,
            metadata: processedImages.metadata
        });
    } catch (error) {
        consoleLog('error', 'Thumbnail download failed', {
            error: error.message,
            thumbnailUrl,
            gameSlug,
            httpStatus: error.response?.status,
            errorCode: error.code
        });
        return null; // Continue without thumbnail
    }
}

/**
 * Downloads and processes game file from API
 * NOTE: API game files are either direct files or ZIP files depending on type
 * @param {string} gameFileUrl - URL of game file
 * @param {string} gameSlug - Game slug for filename
 * @param {string} gameType - Type of game (rom, html, flash, tic80)
 * @param {string} romType - ROM type (if applicable)
 * @returns {Promise<string>} Processed game file path
 */
export async function downloadAndProcessGameFile(gameFileUrl, gameSlug, gameType, romType) {
    try {
        const response = await axios.get(gameFileUrl, {
            responseType: 'arraybuffer',
            timeout: 300000 // 300 seconds for game files
        });
        
        const buffer = Buffer.from(response.data);
        const gameUploadsDir = path.join(process.cwd(), 'uploads', 'games');
        
        // Ensure directory exists
        if (!fs.existsSync(gameUploadsDir)) {
            fs.mkdirSync(gameUploadsDir, { recursive: true });
        }

        // Handle TIC-80 games specially - they come as direct .tic files, not ZIP files
        if (gameType === 'tic80') {
            // Create TIC-80 directory if it doesn't exist
            const tic80Dir = path.join(gameUploadsDir, 'tic80');
            if (!fs.existsSync(tic80Dir)) {
                fs.mkdirSync(tic80Dir, { recursive: true });
            }

            // Save .tic file directly (no ZIP extraction needed)
            const fileName = `game_${gameSlug}_${Date.now()}.tic`;
            const filePath = path.join(tic80Dir, fileName);
            fs.writeFileSync(filePath, buffer);

            consoleLog('info', 'TIC-80 game file saved successfully', {
                fileName,
                fileSize: buffer.length,
                gameSlug
            });

            return `uploads/games/tic80/${fileName}`;
        }

        // Handle Flash games - they come as direct SWF files from /flashmuseum/ endpoints
        if (gameType === 'flash') {
            // Detect if this is a direct SWF file or ZIP containing SWF
            const isSWF = isSWFFile(buffer);
            const isZIP = isZIPFile(buffer);

            let swfBuffer = null;
            let processingType = '';

            if (isSWF) {
                // Direct SWF file - validate and use as-is
                const validation = validateSWFFile(buffer, gameSlug);
                if (!validation.isValid) {
                    throw new Error(`Invalid SWF file: ${validation.errors.join(', ')}`);
                }

                swfBuffer = buffer;
                processingType = 'direct';

                consoleLog('info', 'Direct SWF file detected for Flash game', {
                    gameSlug,
                    fileSize: buffer.length,
                    signature: validation.signature,
                    version: validation.version
                });

            } else if (isZIP) {
                // ZIP file containing SWF - extract it
                consoleLog('info', 'ZIP file detected for Flash game, extracting SWF', {
                    gameSlug,
                    zipSize: buffer.length
                });

                const tempExtractDir = path.join(gameUploadsDir, `temp_flash_extract_${gameSlug}_${Date.now()}`);

                try {
                    // Extract ZIP to temporary directory
                    fs.mkdirSync(tempExtractDir, { recursive: true });
                    await new Promise((resolve, reject) => {
                        unzipper.Open.buffer(buffer)
                            .then(directory => directory.extract({ path: tempExtractDir }))
                            .then(() => resolve())
                            .catch(reject);
                    });

                    // Find SWF file in extracted content
                    const swfFile = findFileInDir(tempExtractDir, '.swf');
                    if (!swfFile) {
                        throw new Error('No SWF file found in the ZIP archive');
                    }

                    // Read and validate SWF file
                    swfBuffer = fs.readFileSync(swfFile);
                    const validation = validateSWFFile(swfBuffer, gameSlug);
                    if (!validation.isValid) {
                        throw new Error(`Invalid SWF file in ZIP: ${validation.errors.join(', ')}`);
                    }

                    processingType = 'extracted';

                    consoleLog('info', 'SWF extracted from ZIP successfully', {
                        gameSlug,
                        originalFileName: path.basename(swfFile),
                        swfSize: swfBuffer.length,
                        signature: validation.signature,
                        version: validation.version
                    });

                } finally {
                    // Clean up temporary extraction directory
                    if (fs.existsSync(tempExtractDir)) {
                        fs.rmSync(tempExtractDir, { recursive: true, force: true });
                    }
                }

            } else {
                throw new Error('Flash game file is neither a valid SWF nor a ZIP file containing SWF');
            }

            // Ensure Flash directory exists
            const flashDir = ensureFlashDirectory(gameUploadsDir);

            // Generate unique filename with timestamp
            const fileName = `game_${gameSlug}_${Date.now()}.swf`;
            const filePath = path.join(flashDir, fileName);

            // Save SWF file to Flash directory
            fs.writeFileSync(filePath, swfBuffer);

            // Verify file was written correctly
            if (!fs.existsSync(filePath)) {
                throw new Error('Failed to save SWF file to disk');
            }

            const savedFileSize = fs.statSync(filePath).size;
            if (savedFileSize !== swfBuffer.length) {
                throw new Error('SWF file size mismatch after saving');
            }

            consoleLog('info', 'Flash game file processed successfully', {
                gameSlug,
                fileName,
                fileSize: savedFileSize,
                relativePath: `uploads/games/flash/${fileName}`,
                processingType
            });

            return `uploads/games/flash/${fileName}`;
        }

        // All other API files are ZIP files - extract them first
        const tempZipPath = path.join(gameUploadsDir, `temp_${gameSlug}_${Date.now()}.zip`);
        const tempExtractDir = path.join(gameUploadsDir, `temp_extract_${gameSlug}_${Date.now()}`);
        
        // Save ZIP file temporarily
        fs.writeFileSync(tempZipPath, buffer);
        
        // Extract ZIP file to temporary directory
        fs.mkdirSync(tempExtractDir, { recursive: true });
        await new Promise((resolve, reject) => {
            unzipper.Open.buffer(buffer)
                .then(directory => directory.extract({ path: tempExtractDir }))
                .then(() => resolve())
                .catch(reject);
        });
        
        // Clean up temporary ZIP file
        fs.unlinkSync(tempZipPath);
        
        let relativePath;
        
        if (gameType === 'html') {
            // For HTML5 games, extract to organized directory structure
            // Create HTML5 directory if it doesn't exist
            const html5Dir = path.join(gameUploadsDir, 'html5');
            if (!fs.existsSync(html5Dir)) {
                fs.mkdirSync(html5Dir, { recursive: true });
            }

            const gameDir = path.join(html5Dir, `${gameSlug}_${Date.now()}`);

            // Move extracted content to final location
            fs.renameSync(tempExtractDir, gameDir);

            // Always point to index.html with organized path
            relativePath = `uploads/games/html5/${path.basename(gameDir)}/index.html`;
        } else if (gameType === 'rom') {
            // For ROM games, detect system from ZIP buffer and store ZIP as-is
            // This follows the original implementation that doesn't extract ROM ZIP files
            const detectedRomSystem = await detectRomSystemFromZipBuffer(buffer);
            const finalRomSystem = romType || detectedRomSystem;
            
            const romSystemDir = path.join(gameUploadsDir, 'roms', finalRomSystem);
            if (!fs.existsSync(romSystemDir)) {
                fs.mkdirSync(romSystemDir, { recursive: true });
            }
            
            // Store the ZIP file as-is (following original implementation)
            const fileName = `game_${gameSlug}_${Date.now()}.zip`;
            const filePath = path.join(romSystemDir, fileName);
            fs.writeFileSync(filePath, buffer);
            relativePath = `uploads/games/roms/${finalRomSystem}/${fileName}`;
            
            // Clean up temporary extraction directory (not needed for ROM)
            fs.rmSync(tempExtractDir, { recursive: true, force: true });
        } else {
            // Clean up temporary extraction directory
            fs.rmSync(tempExtractDir, { recursive: true, force: true });
            throw new Error(`Unsupported game type: ${gameType}`);
        }
        
        return relativePath;
    } catch (error) {
        // Enhanced error handling with Flash-specific cleanup
        consoleLog('error', 'Game file download failed', {
            gameType,
            gameSlug,
            error: error.message,
            errorType: error.name,
            stackTrace: error.stack
        });

        // Clean up any partial Flash files on error
        if (gameType === 'flash') {
            try {
                const flashDir = path.join(process.cwd(), 'uploads', 'games', 'flash');
                if (fs.existsSync(flashDir)) {
                    const partialFiles = fs.readdirSync(flashDir)
                        .filter(file => file.includes(gameSlug))
                        .map(file => path.join(flashDir, file));

                    partialFiles.forEach(file => {
                        try {
                            fs.unlinkSync(file);
                            consoleLog('info', 'Cleaned up partial Flash file', { file });
                        } catch (cleanupError) {
                            consoleLog('warn', 'Failed to cleanup partial Flash file', {
                                file,
                                error: cleanupError.message
                            });
                        }
                    });
                }
            } catch (cleanupError) {
                consoleLog('warn', 'Flash cleanup process failed', {
                    error: cleanupError.message
                });
            }
        }

        throw new Error(`Failed to download game file: ${error.message}`);
    }
}

/**
 * Helper function to find first file with specific extension in directory
 * @param {string} dir - Directory to search
 * @param {string} extension - File extension to look for
 * @returns {string|null} Full path to found file or null
 */
function findFileInDir(dir, extension) {
    const files = fs.readdirSync(dir, { withFileTypes: true });
    
    for (const file of files) {
        const filePath = path.join(dir, file.name);
        
        if (file.isFile() && file.name.toLowerCase().endsWith(extension.toLowerCase())) {
            return filePath;
        } else if (file.isDirectory()) {
            const found = findFileInDir(filePath, extension);
            if (found) return found;
        }
    }
    
    return null;
}

/**
 * Helper function to find first file with any of the specified extensions
 * @param {string} dir - Directory to search
 * @param {string[]} extensions - Array of file extensions to look for
 * @returns {string|null} Full path to found file or null
 */
function findFileInDirByExtensions(dir, extensions) {
    const files = fs.readdirSync(dir, { withFileTypes: true });
    
    for (const file of files) {
        const filePath = path.join(dir, file.name);
        
        if (file.isFile()) {
            const fileExtension = path.extname(file.name).toLowerCase();
            if (extensions.includes(fileExtension)) {
                return filePath;
            }
        } else if (file.isDirectory()) {
            const found = findFileInDirByExtensions(filePath, extensions);
            if (found) return found;
        }
    }
    
    return null;
}