import fs from 'fs';
import path from 'path';
import { fileURLToPath } from 'url';
import { spawn } from 'child_process';
import mysql from 'mysql2/promise';
import { consoleLog } from './logger.js';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const rootDir = path.resolve(__dirname, '../..');
const backupsDir = path.join(rootDir, 'backups', 'migrations');

/**
 * Database backup and safety utilities for migrations
 * Provides backup creation and restoration capabilities
 */

/**
 * Get database configuration from environment
 */
const getDbConfig = () => {
    return {
        host: process.env.DB_HOST || 'localhost',
        port: parseInt(process.env.DB_PORT) || 3306,
        user: process.env.DB_USER || 'root',
        password: process.env.DB_PASSWORD || '',
        database: process.env.DB_DATABASE || 'arcade'
    };
};

/**
 * Ensure backups directory exists
 */
const ensureBackupsDir = () => {
    if (!fs.existsSync(backupsDir)) {
        fs.mkdirSync(backupsDir, { recursive: true });
    }
};

/**
 * Generate backup filename with timestamp
 */
const generateBackupFilename = (prefix = 'pre_migration') => {
    const now = new Date();
    const timestamp = now.toISOString().replace(/[:.]/g, '-').replace('T', '_').split('.')[0];
    return `${prefix}_${timestamp}.sql`;
};

/**
 * Create database backup using mysqldump
 */
const createDatabaseBackup = async (backupName = null) => {
    return new Promise((resolve, reject) => {
        ensureBackupsDir();
        
        const config = getDbConfig();
        const filename = backupName || generateBackupFilename();
        const backupPath = path.join(backupsDir, filename);
        
        const mysqldumpArgs = [
            `--host=${config.host}`,
            `--port=${config.port}`,
            `--user=${config.user}`,
            '--single-transaction',
            '--routines',
            '--triggers',
            '--add-drop-table',
            '--lock-tables=false',
            config.database
        ];
        
        // Add password if provided
        if (config.password) {
            mysqldumpArgs.unshift(`--password=${config.password}`);
        }
        
        const mysqldump = spawn('mysqldump', mysqldumpArgs);
        const writeStream = fs.createWriteStream(backupPath);
        
        mysqldump.stdout.pipe(writeStream);
        
        let errorOutput = '';
        mysqldump.stderr.on('data', (data) => {
            errorOutput += data.toString();
        });
        
        mysqldump.on('close', (code) => {
            writeStream.end();
            
            if (code === 0) {
                const stats = fs.statSync(backupPath);
                resolve({
                    success: true,
                    backupPath,
                    filename,
                    size: stats.size,
                    created: new Date()
                });
            } else {
                // Clean up failed backup file
                if (fs.existsSync(backupPath)) {
                    fs.unlinkSync(backupPath);
                }
                reject(new Error(`mysqldump failed with code ${code}: ${errorOutput}`));
            }
        });
        
        mysqldump.on('error', (error) => {
            writeStream.end();
            reject(new Error(`Failed to execute mysqldump: ${error.message}`));
        });
    });
};

/**
 * Create a logical backup by exporting schema and data separately
 */
const createLogicalBackup = async (backupName = null) => {
    try {
        const config = getDbConfig();
        const connection = await mysql.createConnection(config);
        
        ensureBackupsDir();
        
        const filename = backupName || generateBackupFilename('logical_backup');
        const backupPath = path.join(backupsDir, filename);
        
        let backupContent = '';
        
        // Add header
        backupContent += `-- Logical Database Backup\n`;
        backupContent += `-- Database: ${config.database}\n`;
        backupContent += `-- Created: ${new Date().toISOString()}\n`;
        backupContent += `-- Generated by Migration System\n\n`;
        
        backupContent += `SET NAMES utf8mb4;\n`;
        backupContent += `SET time_zone = '+00:00';\n`;
        backupContent += `SET foreign_key_checks = 0;\n`;
        backupContent += `SET sql_mode = 'NO_AUTO_VALUE_ON_ZERO';\n\n`;
        
        // Get all tables
        const [tables] = await connection.execute('SHOW TABLES');
        
        for (const tableRow of tables) {
            const tableName = Object.values(tableRow)[0];
            
            // Skip migrations table to avoid conflicts
            if (tableName === 'migrations') {
                continue;
            }
            
            backupContent += `-- Table: ${tableName}\n`;
            
            // Get table structure
            const [createTableResult] = await connection.execute(`SHOW CREATE TABLE \`${tableName}\``);
            const createTableSQL = createTableResult[0]['Create Table'];
            
            backupContent += `DROP TABLE IF EXISTS \`${tableName}\`;\n`;
            backupContent += `${createTableSQL};\n\n`;
            
            // Get table data
            const [rows] = await connection.execute(`SELECT * FROM \`${tableName}\``);
            
            if (rows.length > 0) {
                // Get column names
                const columns = Object.keys(rows[0]);
                const columnList = columns.map(col => `\`${col}\``).join(', ');
                
                backupContent += `-- Data for table ${tableName}\n`;
                backupContent += `INSERT INTO \`${tableName}\` (${columnList}) VALUES\n`;
                
                const valueRows = rows.map(row => {
                    const values = columns.map(col => {
                        const value = row[col];
                        if (value === null) return 'NULL';
                        if (typeof value === 'string') return `'${value.replace(/'/g, "''")}'`;
                        if (value instanceof Date) return `'${value.toISOString().slice(0, 19).replace('T', ' ')}'`;
                        return value;
                    });
                    return `(${values.join(', ')})`;
                });
                
                backupContent += valueRows.join(',\n') + ';\n\n';
            }
        }
        
        backupContent += `SET foreign_key_checks = 1;\n`;
        
        await connection.end();
        
        // Write backup to file
        fs.writeFileSync(backupPath, backupContent);
        
        const stats = fs.statSync(backupPath);
        
        return {
            success: true,
            backupPath,
            filename,
            size: stats.size,
            created: new Date(),
            type: 'logical'
        };
        
    } catch (error) {
        throw new Error(`Logical backup failed: ${error.message}`);
    }
};

/**
 * Restore database from backup file
 */
const restoreFromBackup = async (backupPath) => {
    return new Promise((resolve, reject) => {
        if (!fs.existsSync(backupPath)) {
            reject(new Error(`Backup file not found: ${backupPath}`));
            return;
        }
        
        const config = getDbConfig();
        
        const mysqlArgs = [
            `--host=${config.host}`,
            `--port=${config.port}`,
            `--user=${config.user}`,
            config.database
        ];
        
        // Add password if provided
        if (config.password) {
            mysqlArgs.unshift(`--password=${config.password}`);
        }
        
        const mysql = spawn('mysql', mysqlArgs);
        const readStream = fs.createReadStream(backupPath);
        
        readStream.pipe(mysql.stdin);
        
        let errorOutput = '';
        mysql.stderr.on('data', (data) => {
            errorOutput += data.toString();
        });
        
        mysql.on('close', (code) => {
            if (code === 0) {
                resolve({
                    success: true,
                    message: 'Database restored successfully',
                    backupPath
                });
            } else {
                reject(new Error(`mysql restore failed with code ${code}: ${errorOutput}`));
            }
        });
        
        mysql.on('error', (error) => {
            reject(new Error(`Failed to execute mysql restore: ${error.message}`));
        });
    });
};

/**
 * List available backup files
 */
const listBackups = () => {
    ensureBackupsDir();
    
    const files = fs.readdirSync(backupsDir);
    const backups = files
        .filter(file => file.endsWith('.sql'))
        .map(file => {
            const filePath = path.join(backupsDir, file);
            const stats = fs.statSync(filePath);
            
            return {
                filename: file,
                path: filePath,
                size: stats.size,
                created: stats.birthtime,
                modified: stats.mtime
            };
        })
        .sort((a, b) => b.created - a.created); // Sort by creation time, newest first
    
    return backups;
};

/**
 * Delete old backup files (keep only specified number)
 */
const cleanupOldBackups = (keepCount = 10) => {
    const backups = listBackups();
    
    if (backups.length <= keepCount) {
        return {
            deleted: 0,
            kept: backups.length
        };
    }
    
    const toDelete = backups.slice(keepCount);
    let deletedCount = 0;
    
    toDelete.forEach(backup => {
        try {
            fs.unlinkSync(backup.path);
            deletedCount++;
        } catch (error) {
            consoleLog('warn', `Failed to delete backup ${backup.filename}`, { error: error.message });
        }
    });
    
    return {
        deleted: deletedCount,
        kept: backups.length - deletedCount
    };
};

/**
 * Get backup directory size and statistics
 */
const getBackupStats = () => {
    const backups = listBackups();
    
    const totalSize = backups.reduce((sum, backup) => sum + backup.size, 0);
    const oldestBackup = backups.length > 0 ? backups[backups.length - 1].created : null;
    const newestBackup = backups.length > 0 ? backups[0].created : null;
    
    return {
        totalBackups: backups.length,
        totalSize,
        totalSizeMB: Math.round(totalSize / 1024 / 1024 * 100) / 100,
        oldestBackup,
        newestBackup,
        backupsDir
    };
};

/**
 * Verify backup integrity by attempting to parse SQL
 */
const verifyBackupIntegrity = async (backupPath) => {
    try {
        if (!fs.existsSync(backupPath)) {
            return {
                valid: false,
                error: 'Backup file not found'
            };
        }
        
        const content = fs.readFileSync(backupPath, 'utf8');
        
        // Basic checks
        if (content.length === 0) {
            return {
                valid: false,
                error: 'Backup file is empty'
            };
        }
        
        // Check for essential SQL keywords
        const hasCreateTable = content.includes('CREATE TABLE');
        const hasInsert = content.includes('INSERT INTO') || content.includes('No data to insert');
        
        if (!hasCreateTable && !hasInsert) {
            return {
                valid: false,
                error: 'Backup file does not contain expected SQL content'
            };
        }
        
        // Count tables
        const tableMatches = content.match(/CREATE TABLE/g);
        const tableCount = tableMatches ? tableMatches.length : 0;
        
        return {
            valid: true,
            tableCount,
            size: content.length,
            hasData: content.includes('INSERT INTO')
        };
        
    } catch (error) {
        return {
            valid: false,
            error: error.message
        };
    }
};

export {
    createDatabaseBackup,
    createLogicalBackup,
    restoreFromBackup,
    listBackups,
    cleanupOldBackups,
    getBackupStats,
    verifyBackupIntegrity,
    backupsDir
};