import passport from 'passport';
import FacebookStrategy from 'passport-facebook';
import GoogleStrategy from 'passport-google-oauth20';
import { getSetting } from '../models/settings.js';
import { getUserById, createUser } from '../models/users.js';
import { createOrUpdateOAuthAccount, getOAuthAccount, findOAuthAccountByEmail } from '../models/oauth_accounts.js';
import { consoleLog } from './logger.js';
import bcrypt from 'bcrypt';
import i18n from './i18n.js';


/**
 * Initialize Passport.js with OAuth strategies
 */
export const initializePassport = async () => {
    try {
        // Serialize user for session
        passport.serializeUser((user, done) => {
            done(null, user.id);
        });

        // Deserialize user from session
        passport.deserializeUser(async (id, done) => {
            try {
                const users = await getUserById(id);
                const user = users && users.length > 0 ? users[0] : null;
                done(null, user);
            } catch (error) {
                done(error, null);
            }
        });

        // Initialize strategies
        await initializeFacebookStrategy();
        await initializeGoogleStrategy();

        consoleLog('info', 'Passport.js initialized successfully');
        return true;
    } catch (error) {
        consoleLog('error', 'Failed to initialize Passport.js', { error: error.message });
        return false;
    }
};

/**
 * Initialize Facebook OAuth strategy
 */
const initializeFacebookStrategy = async () => {
    try {
        const [enabled, clientId, clientSecret] = await Promise.all([
            getSetting('enable_facebook_login', '0'),
            getSetting('facebook_app_id', ''),
            getSetting('facebook_app_secret', '')
        ]);

        if (enabled !== '1' || !clientId || !clientSecret) {
            consoleLog('info', 'Facebook OAuth disabled or not configured');
            return;
        }

        passport.use(new FacebookStrategy({
            clientID: clientId,
            clientSecret: clientSecret,
            callbackURL: "/auth/facebook/callback",
            profileFields: ['id', 'emails', 'name', 'picture.type(large)']
        }, async (accessToken, refreshToken, profile, done) => {
            try {
                const result = await handleOAuthProfile('facebook', profile, accessToken, refreshToken);
                done(null, result);
            } catch (error) {
                consoleLog('error', 'Facebook OAuth error', { error: error.message });
                done(error, null);
            }
        }));

        consoleLog('info', 'Facebook OAuth strategy initialized');
    } catch (error) {
        consoleLog('error', 'Failed to initialize Facebook strategy', { error: error.message });
    }
};

/**
 * Initialize Google OAuth strategy
 */
const initializeGoogleStrategy = async () => {
    try {
        const [enabled, clientId, clientSecret] = await Promise.all([
            getSetting('enable_google_login', '0'),
            getSetting('google_client_id', ''),
            getSetting('google_client_secret', '')
        ]);

        if (enabled !== '1' || !clientId || !clientSecret) {
            consoleLog('info', 'Google OAuth disabled or not configured');
            return;
        }

        passport.use(new GoogleStrategy({
            clientID: clientId,
            clientSecret: clientSecret,
            callbackURL: "/auth/google/callback"
        }, async (accessToken, refreshToken, profile, done) => {
            try {
                const result = await handleOAuthProfile('google', profile, accessToken, refreshToken);
                done(null, result);
            } catch (error) {
                consoleLog('error', 'Google OAuth error', { error: error.message });
                done(error, null);
            }
        }));

        consoleLog('info', 'Google OAuth strategy initialized');
    } catch (error) {
        consoleLog('error', 'Failed to initialize Google strategy', { error: error.message });
    }
};

/**
 * Handle OAuth profile data and create/link user account
 * @param {string} provider - OAuth provider (facebook, google)
 * @param {Object} profile - OAuth profile data
 * @param {string} accessToken - OAuth access token
 * @param {string} refreshToken - OAuth refresh token
 * @returns {Object} User data or error
 */
const handleOAuthProfile = async (provider, profile, accessToken, refreshToken) => {
    try {
        const providerId = profile.id;
        const email = profile.emails && profile.emails.length > 0 ? profile.emails[0].value : null;
        const firstName = profile.name ? profile.name.givenName : '';
        const lastName = profile.name ? profile.name.familyName : '';
        const displayName = profile.displayName || `${firstName} ${lastName}`.trim();
        const avatarUrl = getProfilePicture(profile, provider);

        if (!email) {
            return { success: false, error: 'Email not provided by OAuth provider' };
        }

        // Check if OAuth account already exists
        let existingOAuthAccount = await getOAuthAccount(provider, providerId);
        
        if (existingOAuthAccount) {
            // Update OAuth account tokens
            await createOrUpdateOAuthAccount({
                userId: existingOAuthAccount.user_id,
                provider,
                providerId,
                providerEmail: email,
                providerName: displayName,
                providerAvatar: avatarUrl,
                accessToken,
                refreshToken,
                expiresAt: null // Set based on provider if needed
            });

            // Return existing user
            const users = await getUserById(existingOAuthAccount.user_id);
            const user = users && users.length > 0 ? users[0] : null;
            
            if (user && user.is_active) {
                consoleLog('info', 'OAuth login successful', { provider, userId: user.id, email });
                return { success: true, user };
            } else {
                return { success: false, error: 'Account is disabled' };
            }
        }

        // Check if user with this email already exists
        const existingUserByEmail = await findOAuthAccountByEmail(email);
        
        if (existingUserByEmail) {
            // Link OAuth account to existing user
            await createOrUpdateOAuthAccount({
                userId: existingUserByEmail.user_id,
                provider,
                providerId,
                providerEmail: email,
                providerName: displayName,
                providerAvatar: avatarUrl,
                accessToken,
                refreshToken,
                expiresAt: null
            });

            const users = await getUserById(existingUserByEmail.user_id);
            const user = users && users.length > 0 ? users[0] : null;
            
            consoleLog('info', 'OAuth account linked to existing user', { provider, userId: user.id, email });
            return { success: true, user };
        }

        // Create new user account
        const username = await generateUniqueUsername(displayName, email);
        const randomPassword = await bcrypt.hash(Math.random().toString(36), 12); // Random password since OAuth user won't use it
        
        // Get default language from settings for new OAuth user
        const defaultLanguage = await i18n.getDefaultLanguage();

        const newUserData = {
            username,
            email,
            password: randomPassword,
            first_name: firstName,
            last_name: lastName,
            user_type: 'user',
            is_verified: 1, // OAuth users are considered verified
            oauth_provider: provider,
            oauth_avatar: avatarUrl,
            language_preference: defaultLanguage
        };

        const userResult = await createUser(newUserData);
        
        if (userResult.success) {
            // Create OAuth account link
            await createOrUpdateOAuthAccount({
                userId: userResult.userId,
                provider,
                providerId,
                providerEmail: email,
                providerName: displayName,
                providerAvatar: avatarUrl,
                accessToken,
                refreshToken,
                expiresAt: null
            });

            const users = await getUserById(userResult.userId);
            const user = users && users.length > 0 ? users[0] : null;
            
            consoleLog('info', 'New user created via OAuth', { provider, userId: user.id, email });
            return { success: true, user, isNewUser: true };
        } else {
            return { success: false, error: 'Failed to create user account' };
        }

    } catch (error) {
        consoleLog('error', 'Error handling OAuth profile', { error: error.message, provider });
        return { success: false, error: error.message };
    }
};

/**
 * Generate unique username from display name and email
 * @param {string} displayName - Display name from OAuth
 * @param {string} email - Email from OAuth
 * @returns {string} Unique username
 */
const generateUniqueUsername = async (displayName, email) => {
    try {
        // Start with display name or email prefix
        let baseUsername = displayName ? 
            displayName.toLowerCase().replace(/[^a-z0-9]/g, '') :
            email.split('@')[0].toLowerCase().replace(/[^a-z0-9]/g, '');
        
        // Ensure minimum length
        if (baseUsername.length < 3) {
            baseUsername = 'user' + baseUsername;
        }
        
        // Truncate if too long
        if (baseUsername.length > 20) {
            baseUsername = baseUsername.substring(0, 20);
        }
        
        let username = baseUsername;
        let counter = 1;
        
        // Check if username exists and append number if needed
        while (await usernameExists(username)) {
            username = baseUsername + counter;
            counter++;
            
            // Prevent infinite loop
            if (counter > 1000) {
                username = baseUsername + Date.now();
                break;
            }
        }
        
        return username;
    } catch (error) {
        // Fallback to timestamp-based username
        return 'user' + Date.now();
    }
};

/**
 * Check if username already exists
 * @param {string} username - Username to check
 * @returns {boolean} Whether username exists
 */
const usernameExists = async (username) => {
    try {
        const { executeQuery } = await import('./mysql.js');
        const query = 'SELECT id FROM users WHERE username = ? LIMIT 1';
        const result = await executeQuery(query, [username]);
        return result.length > 0;
    } catch (error) {
        return false; // Assume doesn't exist on error
    }
};

/**
 * Extract profile picture URL from OAuth profile
 * @param {Object} profile - OAuth profile
 * @param {string} provider - OAuth provider
 * @returns {string|null} Profile picture URL
 */
const getProfilePicture = (profile, provider) => {
    try {
        if (provider === 'facebook') {
            return profile.photos && profile.photos.length > 0 ? profile.photos[0].value : null;
        } else if (provider === 'google') {
            return profile.photos && profile.photos.length > 0 ? profile.photos[0].value : null;
        }
        return null;
    } catch (error) {
        return null;
    }
};

/**
 * Check if OAuth provider is enabled
 * @param {string} provider - OAuth provider (facebook, google)
 * @returns {boolean} Whether provider is enabled
 */
export const isOAuthProviderEnabled = async (provider) => {
    try {
        const enabled = await getSetting(`enable_${provider}_login`, '0');
        return enabled === '1';
    } catch (error) {
        return false;
    }
};

/**
 * Get OAuth provider configuration
 * @param {string} provider - OAuth provider
 * @returns {Object} Provider configuration
 */
export const getOAuthProviderConfig = async (provider) => {
    try {
        if (provider === 'facebook') {
            const [enabled, clientId] = await Promise.all([
                getSetting('enable_facebook_login', '0'),
                getSetting('facebook_app_id', '')
            ]);
            return { enabled: enabled === '1', clientId };
        } else if (provider === 'google') {
            const [enabled, clientId] = await Promise.all([
                getSetting('enable_google_login', '0'),
                getSetting('google_client_id', '')
            ]);
            return { enabled: enabled === '1', clientId };
        }
        return { enabled: false };
    } catch (error) {
        return { enabled: false };
    }
};


export default {
    initializePassport,
    isOAuthProviderEnabled,
    getOAuthProviderConfig
};